<?php
session_start();

// Check if image was uploaded
if (!isset($_SESSION['uploaded_image']) || !file_exists($_SESSION['uploaded_image']['path'])) {
    header('Location: index.php');
    exit;
}

$imageInfo = $_SESSION['uploaded_image'];

// Preset sizes
$presetSizes = [
    '3.5x4.5cm (413x531px)' => ['width' => 413, 'height' => 531],
    '240x320px' => ['width' => 240, 'height' => 320],
    '200x230px' => ['width' => 200, 'height' => 230],
    '350x350px' => ['width' => 350, 'height' => 350],
    '600x600px' => ['width' => 600, 'height' => 600],
    '800x800px' => ['width' => 800, 'height' => 800],
    'Custom' => ['width' => 0, 'height' => 0]
];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Photo Resizer Pro - Edit Your Photo</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 20px;
            padding: 40px;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1);
        }
        
        h1 {
            text-align: center;
            color: #333;
            margin-bottom: 30px;
        }
        
        .main-content {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 40px;
        }
        
        .image-section {
            text-align: center;
        }
        
        .image-container {
            position: relative;
            display: inline-block;
            margin: 20px 0;
            border: 2px solid #ddd;
            border-radius: 8px;
            overflow: hidden;
        }
        
        #sourceImage {
            display: block;
            max-width: 100%;
            max-height: 500px;
            width: auto;
            height: auto;
        }
        
        #cropOverlay {
            position: absolute;
            border: 2px solid #667eea;
            background: rgba(102, 126, 234, 0.1);
            cursor: move;
            display: none;
        }
        
        .handle {
            position: absolute;
            width: 10px;
            height: 10px;
            background: #667eea;
            border: 2px solid white;
        }
        
        .handle-nw { top: -5px; left: -5px; cursor: nw-resize; }
        .handle-ne { top: -5px; right: -5px; cursor: ne-resize; }
        .handle-sw { bottom: -5px; left: -5px; cursor: sw-resize; }
        .handle-se { bottom: -5px; right: -5px; cursor: se-resize; }
        
        .controls-section {
            background: #f8f9fa;
            padding: 30px;
            border-radius: 12px;
        }
        
        .control-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #444;
        }
        
        select, input[type="number"], input[type="range"] {
            width: 100%;
            padding: 10px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 16px;
        }
        
        input[type="range"] {
            padding: 5px;
        }
        
        .preview-section {
            margin-top: 30px;
            text-align: center;
        }
        
        #preview {
            max-width: 300px;
            max-height: 300px;
            border: 2px solid #ddd;
            border-radius: 8px;
        }
        
        .button-group {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 10px;
            margin-top: 20px;
        }
        
        button {
            padding: 12px 20px;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .primary-btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .primary-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
        }
        
        @media (max-width: 768px) {
            .main-content {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>✨ Edit Your Photo</h1>
        
        <div class="main-content">
            <div class="image-section">
                <h3>Original Image</h3>
                <div class="image-container" id="imageContainer">
                    <img id="sourceImage" src="data:image/jpeg;base64,<?php echo base64_encode(file_get_contents($imageInfo['path'])); ?>" alt="Original">
                    <div id="cropOverlay">
                        <div class="handle handle-nw"></div>
                        <div class="handle handle-ne"></div>
                        <div class="handle handle-sw"></div>
                        <div class="handle handle-se"></div>
                    </div>
                </div>
                
                <div class="preview-section">
                    <h3>Preview</h3>
                    <canvas id="preview"></canvas>
                    <p>File size: <span id="fileSize">-</span> KB</p>
                </div>
            </div>
            
            <div class="controls-section">
                <div class="control-group">
                    <label for="presetSize">Size:</label>
                    <select id="presetSize">
                        <?php foreach ($presetSizes as $label => $size): ?>
                            <option value="<?php echo $size['width'] . 'x' . $size['height']; ?>"><?php echo $label; ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="control-group">
                    <label for="brightness">Brightness: <span id="brightnessValue">0</span></label>
                    <input type="range" id="brightness" min="-100" max="100" value="0">
                </div>
                
                <div class="control-group">
                    <label for="contrast">Contrast: <span id="contrastValue">0</span></label>
                    <input type="range" id="contrast" min="-100" max="100" value="0">
                </div>
                
                <div class="control-group">
                    <label for="quality">Quality: <span id="qualityValue">85</span>%</label>
                    <input type="range" id="quality" min="60" max="100" value="85">
                </div>
                
                <div class="button-group">
                    <button class="primary-btn" onclick="updatePreview()">Update Preview</button>
                    <button class="primary-btn" onclick="downloadImage()">Download</button>
                </div>
            </div>
        </div>
    </div>

    <script>
        let crop = { x: 0, y: 0, width: 100, height: 100 };
        let isDragging = false;
        let isResizing = false;
        let currentHandle = null;
        let startPos = { x: 0, y: 0 };
        let imageLoaded = false;
        let img = new Image();
        
        // Initialize
        window.onload = function() {
            const sourceImg = document.getElementById('sourceImage');
            img.onload = function() {
                imageLoaded = true;
                initializeCrop();
                updatePreview();
            };
            img.src = sourceImg.src;
            
            // Event listeners
            setupEventListeners();
        };
        
        function initializeCrop() {
            const container = document.getElementById('imageContainer');
            const sourceImg = document.getElementById('sourceImage');
            const overlay = document.getElementById('cropOverlay');
            
            // Set initial crop to cover most of the image
            crop.width = sourceImg.offsetWidth * 0.8;
            crop.height = sourceImg.offsetHeight * 0.8;
            crop.x = (sourceImg.offsetWidth - crop.width) / 2;
            crop.y = (sourceImg.offsetHeight - crop.height) / 2;
            
            updateCropOverlay();
            overlay.style.display = 'block';
        }
        
        function setupEventListeners() {
            const overlay = document.getElementById('cropOverlay');
            const container = document.getElementById('imageContainer');
            
            // Drag crop area
            overlay.addEventListener('mousedown', function(e) {
                if (e.target === overlay) {
                    isDragging = true;
                    startPos.x = e.clientX - crop.x;
                    startPos.y = e.clientY - crop.y;
                    e.preventDefault();
                }
            });
            
            // Resize handles
            document.querySelectorAll('.handle').forEach(handle => {
                handle.addEventListener('mousedown', function(e) {
                    isResizing = true;
                    currentHandle = this.className.split(' ')[1];
                    startPos.x = e.clientX;
                    startPos.y = e.clientY;
                    startPos.width = crop.width;
                    startPos.height = crop.height;
                    startPos.cropX = crop.x;
                    startPos.cropY = crop.y;
                    e.stopPropagation();
                    e.preventDefault();
                });
            });
            
            // Mouse move
            document.addEventListener('mousemove', function(e) {
                if (isDragging) {
                    const sourceImg = document.getElementById('sourceImage');
                    crop.x = Math.max(0, Math.min(sourceImg.offsetWidth - crop.width, e.clientX - startPos.x));
                    crop.y = Math.max(0, Math.min(sourceImg.offsetHeight - crop.height, e.clientY - startPos.y));
                    updateCropOverlay();
                } else if (isResizing) {
                    handleResize(e);
                }
            });
            
            // Mouse up
            document.addEventListener('mouseup', function() {
                if (isDragging || isResizing) {
                    isDragging = false;
                    isResizing = false;
                    currentHandle = null;
                    updatePreview();
                }
            });
            
            // Control listeners
            document.getElementById('presetSize').addEventListener('change', function() {
                adjustCropAspectRatio();
                updatePreview();
            });
            
            ['brightness', 'contrast', 'quality'].forEach(id => {
                document.getElementById(id).addEventListener('input', function() {
                    document.getElementById(id + 'Value').textContent = this.value;
                    if (id !== 'quality') updatePreview();
                });
            });
        }
        
        function updateCropOverlay() {
            const overlay = document.getElementById('cropOverlay');
            overlay.style.left = crop.x + 'px';
            overlay.style.top = crop.y + 'px';
            overlay.style.width = crop.width + 'px';
            overlay.style.height = crop.height + 'px';
        }
        
        function handleResize(e) {
            const dx = e.clientX - startPos.x;
            const dy = e.clientY - startPos.y;
            const sourceImg = document.getElementById('sourceImage');
            
            switch(currentHandle) {
                case 'handle-se':
                    crop.width = Math.max(50, startPos.width + dx);
                    crop.height = Math.max(50, startPos.height + dy);
                    break;
                case 'handle-sw':
                    crop.width = Math.max(50, startPos.width - dx);
                    crop.height = Math.max(50, startPos.height + dy);
                    crop.x = startPos.cropX + dx;
                    break;
                case 'handle-ne':
                    crop.width = Math.max(50, startPos.width + dx);
                    crop.height = Math.max(50, startPos.height - dy);
                    crop.y = startPos.cropY + dy;
                    break;
                case 'handle-nw':
                    crop.width = Math.max(50, startPos.width - dx);
                    crop.height = Math.max(50, startPos.height - dy);
                    crop.x = startPos.cropX + dx;
                    crop.y = startPos.cropY + dy;
                    break;
            }
            
            // Keep within bounds
            crop.x = Math.max(0, crop.x);
            crop.y = Math.max(0, crop.y);
            if (crop.x + crop.width > sourceImg.offsetWidth) {
                crop.width = sourceImg.offsetWidth - crop.x;
            }
            if (crop.y + crop.height > sourceImg.offsetHeight) {
                crop.height = sourceImg.offsetHeight - crop.y;
            }
            
            updateCropOverlay();
        }
        
        function adjustCropAspectRatio() {
            const preset = document.getElementById('presetSize').value;
            if (preset === '0x0') return;
            
            const [targetWidth, targetHeight] = preset.split('x').map(Number);
            const aspectRatio = targetWidth / targetHeight;
            
            // Adjust crop to maintain aspect ratio
            const newHeight = crop.width / aspectRatio;
            if (newHeight <= document.getElementById('sourceImage').offsetHeight - crop.y) {
                crop.height = newHeight;
            } else {
                crop.height = document.getElementById('sourceImage').offsetHeight - crop.y;
                crop.width = crop.height * aspectRatio;
            }
            
            updateCropOverlay();
        }
        
        function updatePreview() {
            if (!imageLoaded) return;
            
            const canvas = document.getElementById('preview');
            const ctx = canvas.getContext('2d');
            const sourceImg = document.getElementById('sourceImage');
            
            // Get target dimensions
            const preset = document.getElementById('presetSize').value;
            let targetWidth = 300, targetHeight = 300;
            
            if (preset !== '0x0') {
                [targetWidth, targetHeight] = preset.split('x').map(Number);
            }
            
            // Set canvas size
            canvas.width = Math.min(targetWidth, 300);
            canvas.height = Math.min(targetHeight, 300);
            
            // Calculate scale
            const scaleX = sourceImg.naturalWidth / sourceImg.offsetWidth;
            const scaleY = sourceImg.naturalHeight / sourceImg.offsetHeight;
            
            // Apply brightness/contrast
            ctx.filter = `brightness(${100 + parseInt(document.getElementById('brightness').value)}%) contrast(${100 + parseInt(document.getElementById('contrast').value)}%)`;
            
            // Draw cropped image
            ctx.drawImage(
                img,
                crop.x * scaleX,
                crop.y * scaleY,
                crop.width * scaleX,
                crop.height * scaleY,
                0, 0,
                canvas.width,
                canvas.height
            );
            
            // Estimate file size
            const quality = parseInt(document.getElementById('quality').value) / 100;
            canvas.toBlob(function(blob) {
                if (blob) {
                    document.getElementById('fileSize').textContent = (blob.size / 1024).toFixed(1);
                }
            }, 'image/jpeg', quality);
        }
        
        function downloadImage() {
            const canvas = document.getElementById('preview');
            const quality = parseInt(document.getElementById('quality').value) / 100;
            
            canvas.toBlob(function(blob) {
                const url = URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = 'resized_photo.jpg';
                a.click();
                URL.revokeObjectURL(url);
            }, 'image/jpeg', quality);
        }
    </script>
</body>
</html>