<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/error.log');

session_start();
header('Content-Type: application/json');

try {
    // Check if image was uploaded
    if (!isset($_SESSION['uploaded_image'])) {
        throw new Exception('No image found in session. Please upload an image first.');
    }
    
    if (!isset($_SESSION['uploaded_image']['path'])) {
        throw new Exception('Invalid session data - path missing');
    }
    
    if (!file_exists($_SESSION['uploaded_image']['path'])) {
        throw new Exception('Image file not found: ' . $_SESSION['uploaded_image']['path']);
    }

    // Check if it's a POST request
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Invalid request method: ' . $_SERVER['REQUEST_METHOD']);
    }

    // Get parameters with defaults
    $width = isset($_POST['width']) ? intval($_POST['width']) : 0;
    $height = isset($_POST['height']) ? intval($_POST['height']) : 0;
    $cropX = isset($_POST['cropX']) ? intval($_POST['cropX']) : 0;
    $cropY = isset($_POST['cropY']) ? intval($_POST['cropY']) : 0;
    $cropWidth = isset($_POST['cropWidth']) ? intval($_POST['cropWidth']) : 0;
    $cropHeight = isset($_POST['cropHeight']) ? intval($_POST['cropHeight']) : 0;
    $brightness = isset($_POST['brightness']) ? intval($_POST['brightness']) : 0;
    $contrast = isset($_POST['contrast']) ? intval($_POST['contrast']) : 0;
    $background = isset($_POST['background']) ? $_POST['background'] : 'none';
    $quality = isset($_POST['quality']) ? intval($_POST['quality']) : 85;

    // Validate parameters
    if ($width <= 0 || $height <= 0) {
        throw new Exception("Invalid target dimensions: {$width}x{$height}");
    }
    
    if ($cropWidth <= 0 || $cropHeight <= 0) {
        throw new Exception("Invalid crop dimensions: {$cropWidth}x{$cropHeight}");
    }

    // Process the image
    $imagePath = $_SESSION['uploaded_image']['path'];
    
    // Get image info using getimagesize
    $imageInfo = @getimagesize($imagePath);
    if ($imageInfo === false) {
        throw new Exception('Cannot determine image type');
    }
    
    $imageType = $imageInfo[2];
    
    // Create image resource
    switch ($imageType) {
        case IMAGETYPE_JPEG:
            $source = @imagecreatefromjpeg($imagePath);
            break;
        case IMAGETYPE_PNG:
            $source = @imagecreatefrompng($imagePath);
            break;
        case IMAGETYPE_GIF:
            $source = @imagecreatefromgif($imagePath);
            break;
        default:
            throw new Exception('Unsupported image type: ' . $imageType);
    }
    
    if (!$source) {
        throw new Exception('Failed to create image resource from file');
    }
    
    // Get original dimensions
    $origWidth = imagesx($source);
    $origHeight = imagesy($source);
    
    // Validate and adjust crop coordinates
    $cropX = max(0, min($cropX, $origWidth - 1));
    $cropY = max(0, min($cropY, $origHeight - 1));
    $cropWidth = min($cropWidth, $origWidth - $cropX);
    $cropHeight = min($cropHeight, $origHeight - $cropY);
    
    if ($cropWidth <= 0 || $cropHeight <= 0) {
        throw new Exception("Invalid crop area after adjustment: {$cropWidth}x{$cropHeight} at ({$cropX},{$cropY})");
    }
    
    // Create cropped image
    $cropped = @imagecreatetruecolor($cropWidth, $cropHeight);
    if (!$cropped) {
        throw new Exception('Failed to create cropped image');
    }
    
    // Copy the cropped portion
    $copyResult = @imagecopyresampled($cropped, $source, 0, 0, $cropX, $cropY, $cropWidth, $cropHeight, $cropWidth, $cropHeight);
    if (!$copyResult) {
        throw new Exception('Failed to copy cropped area');
    }
    
    // Create final resized image
    $resized = @imagecreatetruecolor($width, $height);
    if (!$resized) {
        throw new Exception('Failed to create resized image');
    }
    
    // Handle background replacement for passport photos
    if ($background !== 'none' && $background !== 'keep') {
        // Resize the cropped image first
        $tempResized = imagecreatetruecolor($width, $height);
        imagecopyresampled($tempResized, $cropped, 0, 0, 0, 0, $width, $height, $cropWidth, $cropHeight);
        
        // Set target background color
        if ($background === 'white') {
            $targetR = 255; $targetG = 255; $targetB = 255;
        } else { // light gray
            $targetR = 240; $targetG = 240; $targetB = 240;
        }
        
        // Fill with target background
        $bgColor = imagecolorallocate($resized, $targetR, $targetG, $targetB);
        imagefill($resized, 0, 0, $bgColor);
        
        // Detect and replace background
        // Sample colors from corners to detect current background
        $corners = [
            [0, 0], [10, 0], [0, 10],
            [$width-1, 0], [$width-11, 0], [$width-1, 10],
            [0, $height-1], [10, $height-1], [0, $height-11],
            [$width-1, $height-1], [$width-11, $height-1], [$width-1, $height-11]
        ];
        
        $bgColors = [];
        foreach ($corners as $corner) {
            if ($corner[0] >= 0 && $corner[0] < $width && $corner[1] >= 0 && $corner[1] < $height) {
                $rgb = imagecolorat($tempResized, $corner[0], $corner[1]);
                $r = ($rgb >> 16) & 0xFF;
                $g = ($rgb >> 8) & 0xFF;
                $b = $rgb & 0xFF;
                $bgColors[] = [$r, $g, $b];
            }
        }
        
        // Average background color
        $avgR = $avgG = $avgB = 0;
        foreach ($bgColors as $color) {
            $avgR += $color[0];
            $avgG += $color[1];
            $avgB += $color[2];
        }
        $count = count($bgColors);
        $avgR = $count > 0 ? $avgR / $count : 255;
        $avgG = $count > 0 ? $avgG / $count : 255;
        $avgB = $count > 0 ? $avgB / $count : 255;
        
        // Copy non-background pixels
        for ($y = 0; $y < $height; $y++) {
            for ($x = 0; $x < $width; $x++) {
                $rgb = imagecolorat($tempResized, $x, $y);
                $r = ($rgb >> 16) & 0xFF;
                $g = ($rgb >> 8) & 0xFF;
                $b = $rgb & 0xFF;
                
                // Calculate color distance from detected background
                $distance = sqrt(pow($r - $avgR, 2) + pow($g - $avgG, 2) + pow($b - $avgB, 2));
                
                // If color is significantly different from background, copy it
                if ($distance > 30) { // Threshold for background detection
                    $color = imagecolorallocate($resized, $r, $g, $b);
                    imagesetpixel($resized, $x, $y, $color);
                }
            }
        }
        
        imagedestroy($tempResized);
    } else {
        // No background replacement or keep original background
        if ($background === 'keep' || $background === 'none') {
            // For transparent images, preserve transparency
            if ($imageType === IMAGETYPE_PNG) {
                imagealphablending($resized, false);
                imagesavealpha($resized, true);
                $transparent = imagecolorallocatealpha($resized, 255, 255, 255, 127);
                imagefilledrectangle($resized, 0, 0, $width, $height, $transparent);
                imagealphablending($resized, true);
            }
        }
        
        // Just resize normally
        imagecopyresampled($resized, $cropped, 0, 0, 0, 0, $width, $height, $cropWidth, $cropHeight);
    }
    
    // Apply brightness/contrast
    if ($brightness !== 0) {
        imagefilter($resized, IMG_FILTER_BRIGHTNESS, $brightness);
    }
    if ($contrast !== 0) {
        imagefilter($resized, IMG_FILTER_CONTRAST, -$contrast);
    }
    
    // Create output buffer
    ob_start();
    imagejpeg($resized, null, $quality);
    $imageData = ob_get_clean();
    
    // Calculate file size
    $fileSize = strlen($imageData);
    
    // Clean up
    imagedestroy($source);
    imagedestroy($cropped);
    imagedestroy($resized);
    
    // Return success response
    echo json_encode([
        'success' => true,
        'data' => 'data:image/jpeg;base64,' . base64_encode($imageData),
        'size' => round($fileSize / 1024, 1)
    ]);
    
} catch (Exception $e) {
    // Log the error
    error_log('Process Image Error: ' . $e->getMessage() . ' in ' . $e->getFile() . ':' . $e->getLine());
    
    // Return error response
    echo json_encode([
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'post' => $_POST,
            'session' => isset($_SESSION['uploaded_image']) ? 'set' : 'not set'
        ]
    ]);
}
?>