<?php
session_start();

// Include rate limiter and config
require_once __DIR__ . '/includes/RateLimiter.php';
require_once __DIR__ . '/includes/IPRateLimiter.php';
require_once __DIR__ . '/includes/ServerMonitor.php';
require_once __DIR__ . '/includes/config.php';

// Configuration
$maxFileSize = 5 * 1024 * 1024; // 5MB
$allowedTypes = ['image/png', 'image/jpeg', 'image/jpg', 'image/gif', 'image/bmp', 'image/webp'];
$allowedExtensions = ['png', 'jpg', 'jpeg', 'gif', 'bmp', 'webp'];

// Check rate limit and server status
$rateLimiter = new RateLimiter(2, 60);
$remainingRequests = $rateLimiter->getRemainingRequests('image_process');

$ipLimiter = new IPRateLimiter(2, 5);
$ipRemaining = $ipLimiter->getRemainingRequests();
$showCaptcha = $ipLimiter->isSuspicious() || $ipRemaining['minute'] == 0 || $ipRemaining['hour'] == 0;

$monitor = new ServerMonitor();
$serverHealthy = $monitor->checkResources();

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['image'])) {
    try {
        // Verify CAPTCHA if enabled
        if (CAPTCHA_ENABLED && $showCaptcha) {
            if (!isset($_POST['g-recaptcha-response']) || empty($_POST['g-recaptcha-response'])) {
                throw new Exception('Please complete the CAPTCHA verification');
            }
            
            // Verify with Google
            $recaptcha_url = 'https://www.google.com/recaptcha/api/siteverify';
            $recaptcha_data = [
                'secret' => CAPTCHA_SECRET_KEY,
                'response' => $_POST['g-recaptcha-response'],
                'remoteip' => $ipLimiter->getUserIP()
            ];
            
            $recaptcha_options = [
                'http' => [
                    'method' => 'POST',
                    'header' => 'Content-Type: application/x-www-form-urlencoded',
                    'content' => http_build_query($recaptcha_data)
                ]
            ];
            
            $recaptcha_context = stream_context_create($recaptcha_options);
            $recaptcha_result = @file_get_contents($recaptcha_url, false, $recaptcha_context);
            
            if ($recaptcha_result) {
                $recaptcha_json = json_decode($recaptcha_result, true);
                if (!$recaptcha_json['success']) {
                    throw new Exception('CAPTCHA verification failed. Please try again.');
                }
            } else {
                throw new Exception('Could not verify CAPTCHA. Please try again.');
            }
        }
        
        // Validate file upload
        if ($_FILES['image']['error'] !== UPLOAD_ERR_OK) {
            throw new Exception('File upload failed');
        }
        
        // Check file size
        if ($_FILES['image']['size'] > $maxFileSize) {
            throw new Exception('File size exceeds 5MB limit');
        }
        
        // Check file type
        $fileType = $_FILES['image']['type'];
        if (!in_array($fileType, $allowedTypes)) {
            throw new Exception('Invalid file type. Allowed types: ' . implode(', ', $allowedExtensions));
        }
        
        // Get image info
        $imageInfo = getimagesize($_FILES['image']['tmp_name']);
        if (!$imageInfo) {
            throw new Exception('Invalid image file');
        }
        
        // Save uploaded file temporarily
        $uploadDir = __DIR__ . '/temp/';
        if (!file_exists($uploadDir)) {
            mkdir($uploadDir, 0777, true);
        }
        
        // Clean up old files (older than 1 hour)
        $files = glob($uploadDir . '*');
        foreach ($files as $file) {
            if (is_file($file) && time() - filemtime($file) > 3600) {
                unlink($file);
            }
        }
        
        $tempFile = $uploadDir . session_id() . '_' . uniqid() . '_' . basename($_FILES['image']['name']);
        if (!move_uploaded_file($_FILES['image']['tmp_name'], $tempFile)) {
            throw new Exception('Failed to save uploaded file');
        }
        
        // Store file info in session
        $_SESSION['uploaded_image'] = [
            'path' => $tempFile,
            'name' => $_FILES['image']['name'],
            'type' => $fileType,
            'width' => $imageInfo[0],
            'height' => $imageInfo[1]
        ];
        
        // Redirect to processing page
        header('Location: process.php');
        exit;
        
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Photo Resizer Pro - Professional Image Optimization</title>
    <?php if (CAPTCHA_ENABLED): ?>
    <script src="https://www.google.com/recaptcha/api.js" async defer></script>
    <?php endif; ?>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .container {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1);
            padding: 40px;
            max-width: 500px;
            width: 100%;
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .header {
            text-align: center;
            margin-bottom: 40px;
        }
        
        .logo {
            width: 80px;
            height: 80px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 20px;
            font-size: 40px;
            color: white;
            box-shadow: 0 10px 20px rgba(102, 126, 234, 0.3);
        }
        
        h1 {
            color: #333;
            font-size: 28px;
            font-weight: 700;
            margin-bottom: 10px;
        }
        
        .subtitle {
            color: #666;
            font-size: 16px;
            line-height: 1.5;
        }
        
        .upload-area {
            border: 2px dashed #ddd;
            border-radius: 16px;
            padding: 40px 20px;
            text-align: center;
            transition: all 0.3s ease;
            background: #fafafa;
            position: relative;
            overflow: hidden;
        }
        
        .upload-area:hover {
            border-color: #667eea;
            background: #f5f3ff;
        }
        
        .upload-area.dragover {
            border-color: #667eea;
            background: #f5f3ff;
            transform: scale(1.02);
        }
        
        .upload-icon {
            font-size: 60px;
            color: #667eea;
            margin-bottom: 20px;
            display: block;
        }
        
        .upload-text {
            color: #333;
            font-size: 18px;
            font-weight: 500;
            margin-bottom: 10px;
        }
        
        .upload-subtext {
            color: #666;
            font-size: 14px;
            margin-bottom: 20px;
        }
        
        input[type="file"] {
            display: none;
        }
        
        .upload-button {
            display: inline-block;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 12px 30px;
            border-radius: 30px;
            font-size: 16px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s ease;
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
        }
        
        .upload-button:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 20px rgba(102, 126, 234, 0.4);
        }
        
        .file-info {
            margin-top: 20px;
            padding: 20px;
            background: #e8f5e9;
            border: 2px solid #4caf50;
            border-radius: 12px;
            display: none;
            animation: fadeIn 0.3s ease;
            position: relative;
        }
        
        .file-info::before {
            content: "✓";
            position: absolute;
            top: -15px;
            right: 20px;
            background: #4caf50;
            color: white;
            width: 30px;
            height: 30px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 18px;
            font-weight: bold;
        }
        
        @keyframes fadeIn {
            from {
                opacity: 0;
                transform: translateY(10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .file-name {
            color: #333;
            font-weight: 500;
            margin-bottom: 5px;
            word-break: break-word;
        }
        
        .file-size {
            color: #666;
            font-size: 14px;
        }
        
        .submit-button {
            width: 100%;
            background: linear-gradient(135deg, #48c774 0%, #3ec46d 100%);
            color: white;
            padding: 18px;
            border: none;
            border-radius: 12px;
            font-size: 20px;
            font-weight: 600;
            cursor: pointer;
            margin-top: 20px;
            transition: all 0.3s ease;
            box-shadow: 0 5px 15px rgba(72, 199, 116, 0.3);
            display: none;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0% {
                box-shadow: 0 5px 15px rgba(72, 199, 116, 0.3);
            }
            50% {
                box-shadow: 0 5px 25px rgba(72, 199, 116, 0.5);
            }
            100% {
                box-shadow: 0 5px 15px rgba(72, 199, 116, 0.3);
            }
        }
        
        .submit-button:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 8px 20px rgba(72, 199, 116, 0.4);
        }
        
        .submit-button:disabled {
            background: #ccc;
            cursor: not-allowed;
            box-shadow: none;
        }
        
        .error {
            background: #fee;
            color: #c33;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            text-align: center;
            display: block;
            animation: shake 0.5s ease;
        }
        
        @keyframes shake {
            0%, 100% { transform: translateX(0); }
            25% { transform: translateX(-10px); }
            75% { transform: translateX(10px); }
        }
        
        .upload-success {
            display: none;
            text-align: center;
            margin-top: 20px;
            animation: fadeIn 0.5s ease;
        }
        
        .success-message {
            background: #4caf50;
            color: white;
            padding: 15px 25px;
            border-radius: 50px;
            display: inline-block;
            font-weight: 600;
            margin-bottom: 10px;
        }
        
        .features {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-top: 40px;
        }
        
        .feature {
            text-align: center;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 12px;
            transition: all 0.3s ease;
        }
        
        .feature:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.1);
        }
        
        .feature-icon {
            font-size: 30px;
            color: #667eea;
            margin-bottom: 10px;
        }
        
        .feature-title {
            font-weight: 600;
            color: #333;
            margin-bottom: 5px;
        }
        
        .feature-desc {
            font-size: 14px;
            color: #666;
        }
        
        /* New informative sections */
        .info-section, .requirements-section, .tips-section {
            margin-top: 60px;
            padding-top: 40px;
            border-top: 1px solid rgba(0, 0, 0, 0.1);
        }
        
        h2 {
            color: #333;
            font-size: 24px;
            font-weight: 700;
            text-align: center;
            margin-bottom: 40px;
        }
        
        .steps {
            display: grid;
            gap: 30px;
            margin-top: 30px;
        }
        
        .step {
            display: flex;
            gap: 20px;
            align-items: flex-start;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 12px;
            transition: all 0.3s ease;
        }
        
        .step:hover {
            transform: translateX(10px);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }
        
        .step-number {
            flex-shrink: 0;
            width: 50px;
            height: 50px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            font-weight: bold;
        }
        
        .step-content h3 {
            color: #333;
            margin-bottom: 10px;
            font-size: 18px;
        }
        
        .step-content p {
            color: #666;
            line-height: 1.6;
        }
        
        .req-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
        }
        
        .req-item {
            background: #f8f9fa;
            padding: 25px;
            border-radius: 12px;
            border: 1px solid #e0e0e0;
            transition: all 0.3s ease;
        }
        
        .req-item:hover {
            border-color: #667eea;
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.1);
        }
        
        .req-item h4 {
            color: #333;
            margin-bottom: 15px;
            font-size: 18px;
        }
        
        .req-item ul {
            list-style: none;
            padding: 0;
        }
        
        .req-item li {
            color: #666;
            padding: 5px 0;
            font-size: 14px;
        }
        
        .req-item li:before {
            content: "✓ ";
            color: #667eea;
            font-weight: bold;
        }
        
        .tips-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
        }
        
        .tip {
            display: flex;
            gap: 15px;
            align-items: flex-start;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 12px;
            transition: all 0.3s ease;
        }
        
        .tip:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }
        
        .tip-icon {
            font-size: 30px;
            flex-shrink: 0;
        }
        
        .tip p {
            color: #666;
            line-height: 1.5;
            font-size: 14px;
        }
        
        .tip strong {
            color: #333;
        }
        
        @media (max-width: 600px) {
            .container {
                padding: 30px 20px;
            }
            
            h1 {
                font-size: 24px;
            }
            
            h2 {
                font-size: 20px;
            }
            
            .upload-area {
                padding: 30px 15px;
            }
            
            .features {
                grid-template-columns: 1fr;
                gap: 15px;
            }
            
            .step {
                flex-direction: column;
                text-align: center;
            }
            
            .req-grid, .tips-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div class="logo">📸</div>
            <h1>Photo Resizer Pro</h1>
            <p class="subtitle">Professional passport photo optimization for official documents</p>
        </div>
        
        <?php if (!$serverHealthy): ?>
            <div class="error" style="margin-bottom: 20px;">
                ⚠️ Server is currently experiencing high load. Processing may be slower than usual.
            </div>
        <?php endif; ?>
        
        <?php if ($remainingRequests < 2): ?>
            <div class="error" style="margin-bottom: 20px; background: #fff3cd; color: #856404;">
                ⏱️ Session limit: 
                <?php if ($remainingRequests == 0): ?>
                    You've used all your processing requests for this minute. Please wait before uploading more images.
                <?php else: ?>
                    You have only <?php echo $remainingRequests; ?> processing request remaining this minute. Use it wisely!
                <?php endif; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($ipRemaining['minute'] == 0 || $ipRemaining['hour'] == 0): ?>
            <div class="error" style="margin-bottom: 20px; background: #ffebee; color: #c62828;">
                🚫 Rate limit reached: 
                <?php if ($ipRemaining['minute'] == 0 && $ipRemaining['hour'] == 0): ?>
                    You've exceeded both minute and hour limits. Please wait before uploading more images.
                <?php elseif ($ipRemaining['minute'] == 0): ?>
                    You've made too many requests this minute. Please wait a moment.
                <?php else: ?>
                    You've reached your hourly limit of 5 requests. Please try again later.
                <?php endif; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($showCaptcha): ?>
            <div class="error" style="margin-bottom: 20px; background: #e3f2fd; color: #1565c0;">
                🔐 Security check required: Please complete the CAPTCHA below to continue.
            </div>
        <?php endif; ?>
        
        <?php if (isset($error)): ?>
            <div class="error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <form method="POST" enctype="multipart/form-data" id="uploadForm">
            <div class="upload-area" id="uploadArea">
                <span class="upload-icon">⬆️</span>
                <p class="upload-text">Drop your photo here</p>
                <p class="upload-subtext">or click to browse</p>
                <label for="image" class="upload-button">Choose Photo</label>
                <input type="file" id="image" name="image" accept="image/*" required>
            </div>
            
            <div class="file-info" id="fileInfo">
                <div class="file-name" id="fileName"></div>
                <div class="file-size" id="fileSize"></div>
            </div>
            
            <div class="upload-success" id="uploadSuccess">
                <div class="success-message">✓ Photo uploaded successfully!</div>
            </div>
            
            <?php if (CAPTCHA_ENABLED && $showCaptcha): ?>
            <div style="margin: 20px 0; display: flex; justify-content: center;">
                <div class="g-recaptcha" data-sitekey="<?php echo CAPTCHA_SITE_KEY; ?>"></div>
            </div>
            <?php endif; ?>
            
            <button type="submit" class="submit-button" id="submitButton">
                Process Photo →
            </button>
        </form>
        
        <div class="features">
            <div class="feature">
                <div class="feature-icon">✨</div>
                <div class="feature-title">Smart Resize</div>
                <div class="feature-desc">Multiple preset sizes for official documents</div>
            </div>
            <div class="feature">
                <div class="feature-icon">🎨</div>
                <div class="feature-title">Photo Enhancement</div>
                <div class="feature-desc">Adjust brightness, contrast & background</div>
            </div>
            <div class="feature">
                <div class="feature-icon">📏</div>
                <div class="feature-title">Size Optimization</div>
                <div class="feature-desc">Meet exact file size requirements</div>
            </div>
        </div>
        
        <div class="info-section">
            <h2>📚 How It Works</h2>
            <div class="steps">
                <div class="step">
                    <div class="step-number">1</div>
                    <div class="step-content">
                        <h3>Upload Your Photo</h3>
                        <p>Drag and drop or click to select your image. We support PNG, JPG, JPEG, GIF, BMP, and WebP formats up to 5MB.</p>
                    </div>
                </div>
                <div class="step">
                    <div class="step-number">2</div>
                    <div class="step-content">
                        <h3>Choose Size & Edit</h3>
                        <p>Select from preset passport/visa sizes or enter custom dimensions. Crop, adjust brightness, contrast, and background color.</p>
                    </div>
                </div>
                <div class="step">
                    <div class="step-number">3</div>
                    <div class="step-content">
                        <h3>Download Optimized Photo</h3>
                        <p>Get your photo in JPG format with proper DPI settings. File size automatically optimized to meet official requirements (20-100KB).</p>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="requirements-section">
            <h2>📋 Common Requirements</h2>
            <div class="req-grid">
                <div class="req-item">
                    <h4>🇺🇸 US Passport</h4>
                    <ul>
                        <li>Size: 2x2 inches (600x600px)</li>
                        <li>File: 20-240KB</li>
                        <li>Format: JPG</li>
                        <li>Background: White</li>
                    </ul>
                </div>
                <div class="req-item">
                    <h4>🇮🇳 Indian Passport</h4>
                    <ul>
                        <li>Size: 3.5x4.5cm (413x531px)</li>
                        <li>File: 20-300KB</li>
                        <li>Format: JPG</li>
                        <li>Background: White</li>
                    </ul>
                </div>
                <div class="req-item">
                    <h4>🇪🇺 Schengen Visa</h4>
                    <ul>
                        <li>Size: 35x45mm (413x531px)</li>
                        <li>File: 20-100KB</li>
                        <li>Format: JPG</li>
                        <li>Background: Light gray</li>
                    </ul>
                </div>
                <div class="req-item">
                    <h4>📄 Other Documents</h4>
                    <ul>
                        <li>ID Cards: 240x320px</li>
                        <li>LinkedIn: 400x400px</li>
                        <li>Forms: 200x230px</li>
                        <li>Custom sizes available</li>
                    </ul>
                </div>
            </div>
        </div>
        
        <div class="tips-section">
            <h2>💡 Photo Tips</h2>
            <div class="tips-grid">
                <div class="tip">
                    <span class="tip-icon">📸</span>
                    <p><strong>Good Lighting:</strong> Use natural light or bright indoor lighting. Avoid shadows on face.</p>
                </div>
                <div class="tip">
                    <span class="tip-icon">👤</span>
                    <p><strong>Neutral Expression:</strong> Keep a neutral facial expression with both eyes open.</p>
                </div>
                <div class="tip">
                    <span class="tip-icon">👔</span>
                    <p><strong>Proper Attire:</strong> Wear formal clothing that contrasts with the background.</p>
                </div>
                <div class="tip">
                    <span class="tip-icon">🎯</span>
                    <p><strong>Center Face:</strong> Face should be centered and occupy 70-80% of the photo.</p>
                </div>
            </div>
        </div>
    </div>

    <script>
        const uploadArea = document.getElementById('uploadArea');
        const fileInput = document.getElementById('image');
        const fileInfo = document.getElementById('fileInfo');
        const fileName = document.getElementById('fileName');
        const fileSize = document.getElementById('fileSize');
        const submitButton = document.getElementById('submitButton');
        const uploadForm = document.getElementById('uploadForm');
        const uploadSuccess = document.getElementById('uploadSuccess');
        
        // Drag and drop functionality
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            uploadArea.addEventListener(eventName, preventDefaults, false);
            document.body.addEventListener(eventName, preventDefaults, false);
        });
        
        function preventDefaults(e) {
            e.preventDefault();
            e.stopPropagation();
        }
        
        ['dragenter', 'dragover'].forEach(eventName => {
            uploadArea.addEventListener(eventName, highlight, false);
        });
        
        ['dragleave', 'drop'].forEach(eventName => {
            uploadArea.addEventListener(eventName, unhighlight, false);
        });
        
        function highlight(e) {
            uploadArea.classList.add('dragover');
        }
        
        function unhighlight(e) {
            uploadArea.classList.remove('dragover');
        }
        
        uploadArea.addEventListener('drop', handleDrop, false);
        
        function handleDrop(e) {
            const dt = e.dataTransfer;
            const files = dt.files;
            
            if (files.length > 0) {
                fileInput.files = files;
                handleFile(files[0]);
            }
        }
        
        // File input change
        fileInput.addEventListener('change', function(e) {
            const file = e.target.files[0];
            if (file) {
                handleFile(file);
            }
        });
        
        function handleFile(file) {
            const maxSize = 5 * 1024 * 1024; // 5MB
            const uploadSuccess = document.getElementById('uploadSuccess');
            
            if (file.size > maxSize) {
                alert('File size exceeds 5MB limit');
                fileInput.value = '';
                fileInfo.style.display = 'none';
                submitButton.style.display = 'none';
                uploadSuccess.style.display = 'none';
                return;
            }
            
            // Validate file type
            const validTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/bmp', 'image/webp'];
            if (!validTypes.includes(file.type)) {
                alert('Please select a valid image file (JPG, PNG, GIF, BMP, or WebP)');
                fileInput.value = '';
                fileInfo.style.display = 'none';
                submitButton.style.display = 'none';
                uploadSuccess.style.display = 'none';
                return;
            }
            
            // Show file info
            fileName.textContent = `📷 ${file.name}`;
            fileSize.textContent = `Size: ${(file.size / 1024).toFixed(1)} KB`;
            fileInfo.style.display = 'block';
            uploadSuccess.style.display = 'block';
            submitButton.style.display = 'block';
            
            // Update upload area appearance
            uploadArea.style.borderColor = '#4caf50';
            uploadArea.style.background = '#f1f8f4';
            
            // Scroll to submit button
            setTimeout(() => {
                submitButton.scrollIntoView({ behavior: 'smooth', block: 'center' });
            }, 300);
            
            // Add preview if possible
            if (file.type.startsWith('image/')) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    // You can add image preview here if needed
                };
                reader.readAsDataURL(file);
            }
        }
        
        // Form submission animation
        uploadForm.addEventListener('submit', function() {
            submitButton.disabled = true;
            submitButton.textContent = 'Processing...';
        });
    </script>
</body>
</html>