<?php
class ProcessingQueue {
    private $queueDir;
    private $maxConcurrent;
    private $processTimeout;
    
    public function __construct($maxConcurrent = 25, $processTimeout = 30) {
        $this->queueDir = __DIR__ . '/../temp/queue/';
        $this->maxConcurrent = $maxConcurrent;
        $this->processTimeout = $processTimeout;
        
        // Create queue directory if it doesn't exist
        if (!file_exists($this->queueDir)) {
            mkdir($this->queueDir, 0777, true);
        }
        
        // Clean up stale processes
        $this->cleanupStaleProcesses();
    }
    
    public function addToQueue($sessionId, $data) {
        $queueId = uniqid('queue_', true);
        $queueFile = $this->queueDir . $queueId . '.json';
        
        $queueData = [
            'id' => $queueId,
            'session_id' => $sessionId,
            'status' => 'pending',
            'created_at' => time(),
            'data' => $data
        ];
        
        file_put_contents($queueFile, json_encode($queueData));
        return $queueId;
    }
    
    public function canProcess() {
        $activeCount = $this->getActiveProcessCount();
        return $activeCount < $this->maxConcurrent;
    }
    
    public function startProcessing($queueId) {
        $queueFile = $this->queueDir . $queueId . '.json';
        
        if (!file_exists($queueFile)) {
            return false;
        }
        
        $data = json_decode(file_get_contents($queueFile), true);
        $data['status'] = 'processing';
        $data['started_at'] = time();
        
        file_put_contents($queueFile, json_encode($data));
        return true;
    }
    
    public function completeProcessing($queueId) {
        $queueFile = $this->queueDir . $queueId . '.json';
        
        if (file_exists($queueFile)) {
            @unlink($queueFile);
        }
    }
    
    public function getQueuePosition($queueId) {
        $files = glob($this->queueDir . '*.json');
        $pendingQueue = [];
        
        foreach ($files as $file) {
            $data = json_decode(file_get_contents($file), true);
            if ($data['status'] === 'pending') {
                $pendingQueue[$data['id']] = $data['created_at'];
            }
        }
        
        asort($pendingQueue);
        $position = 1;
        
        foreach ($pendingQueue as $id => $timestamp) {
            if ($id === $queueId) {
                return $position;
            }
            $position++;
        }
        
        return 0;
    }
    
    public function getStatus($queueId) {
        $queueFile = $this->queueDir . $queueId . '.json';
        
        if (!file_exists($queueFile)) {
            return ['status' => 'completed'];
        }
        
        $data = json_decode(file_get_contents($queueFile), true);
        return [
            'status' => $data['status'],
            'position' => $this->getQueuePosition($queueId),
            'active_processes' => $this->getActiveProcessCount()
        ];
    }
    
    private function getActiveProcessCount() {
        $files = glob($this->queueDir . '*.json');
        $count = 0;
        
        foreach ($files as $file) {
            $data = json_decode(file_get_contents($file), true);
            if ($data['status'] === 'processing') {
                $count++;
            }
        }
        
        return $count;
    }
    
    private function cleanupStaleProcesses() {
        $files = glob($this->queueDir . '*.json');
        $now = time();
        
        foreach ($files as $file) {
            $data = json_decode(file_get_contents($file), true);
            
            // Remove old pending items (> 5 minutes)
            if ($data['status'] === 'pending' && ($now - $data['created_at']) > 300) {
                @unlink($file);
                continue;
            }
            
            // Reset stale processing items
            if ($data['status'] === 'processing' && 
                isset($data['started_at']) && 
                ($now - $data['started_at']) > $this->processTimeout) {
                $data['status'] = 'pending';
                unset($data['started_at']);
                file_put_contents($file, json_encode($data));
            }
        }
    }
}
?>