<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/includes/RateLimiter.php';
require_once __DIR__ . '/includes/IPRateLimiter.php';

// Function to display test results
function displayResult($test, $result, $details = '') {
    $status = $result ? '✅ PASS' : '❌ FAIL';
    $color = $result ? 'green' : 'red';
    echo "<div style='margin: 10px 0; padding: 10px; background: #f5f5f5; border-left: 4px solid $color;'>";
    echo "<strong style='color: $color;'>$status</strong> - $test";
    if ($details) {
        echo "<br><small style='color: #666;'>$details</small>";
    }
    echo "</div>";
}

// Function to simulate time passing
function simulateTime(&$rateLimiter, $seconds) {
    // We'll modify the internal state by directly accessing the cache files
    // This is a hack for testing purposes only
    $reflection = new ReflectionClass($rateLimiter);
    $cacheDir = $reflection->getProperty('cacheDir');
    $cacheDir->setAccessible(true);
    $cacheDirPath = $cacheDir->getValue($rateLimiter);
    
    $files = glob($cacheDirPath . '*.json');
    foreach ($files as $file) {
        $data = json_decode(file_get_contents($file), true);
        if (is_array($data)) {
            // Subtract seconds from all timestamps
            $newData = array_map(function($timestamp) use ($seconds) {
                return $timestamp - $seconds;
            }, $data);
            file_put_contents($file, json_encode($newData));
        }
    }
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>Rate Limiter Test Suite</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 1200px;
            margin: 20px auto;
            padding: 20px;
            background: #f0f0f0;
        }
        .test-section {
            background: white;
            padding: 20px;
            margin: 20px 0;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        h2 {
            color: #333;
            border-bottom: 2px solid #667eea;
            padding-bottom: 10px;
        }
        .controls {
            margin: 20px 0;
            padding: 15px;
            background: #e8f4f8;
            border-radius: 8px;
        }
        button {
            padding: 10px 20px;
            margin: 5px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            background: #667eea;
            color: white;
        }
        button:hover {
            background: #5a67d8;
        }
        pre {
            background: #f5f5f5;
            padding: 10px;
            border-radius: 4px;
            overflow-x: auto;
        }
        .warning {
            background: #fff3cd;
            border: 1px solid #ffeaa7;
            padding: 10px;
            border-radius: 4px;
            margin: 10px 0;
        }
    </style>
</head>
<body>
    <h1>🧪 Rate Limiter Test Suite</h1>
    
    <div class="warning">
        <strong>⚠️ Warning:</strong> This test page is for development only. It should be removed in production.
    </div>

    <div class="controls">
        <h3>Test Controls</h3>
        <button onclick="location.reload()">🔄 Reset All Tests</button>
        <button onclick="clearAllLimits()">🗑️ Clear All Rate Limits</button>
    </div>

    <?php
    // Test 1: Session Rate Limiter Basic Functionality
    echo "<div class='test-section'>";
    echo "<h2>Test 1: Session Rate Limiter (2 requests per 60 seconds)</h2>";
    
    $rateLimiter = new RateLimiter(2, 60);
    
    // Test initial state
    $remaining = $rateLimiter->getRemainingRequests('test1');
    displayResult("Initial remaining requests should be 2", $remaining == 2, "Remaining: $remaining");
    
    // First request
    $allowed1 = $rateLimiter->isAllowed('test1');
    displayResult("First request should be allowed", $allowed1);
    
    $remaining = $rateLimiter->getRemainingRequests('test1');
    displayResult("Remaining requests should be 1", $remaining == 1, "Remaining: $remaining");
    
    // Second request
    $allowed2 = $rateLimiter->isAllowed('test1');
    displayResult("Second request should be allowed", $allowed2);
    
    $remaining = $rateLimiter->getRemainingRequests('test1');
    displayResult("Remaining requests should be 0", $remaining == 0, "Remaining: $remaining");
    
    // Third request (should be blocked)
    $allowed3 = $rateLimiter->isAllowed('test1');
    displayResult("Third request should be blocked", !$allowed3);
    
    // Test wait time
    $waitTime = $rateLimiter->getWaitTime('test1');
    displayResult("Wait time should be between 1-60 seconds", $waitTime > 0 && $waitTime <= 60, "Wait time: $waitTime seconds");
    
    echo "</div>";

    // Test 2: IP Rate Limiter
    echo "<div class='test-section'>";
    echo "<h2>Test 2: IP Rate Limiter (2 per minute, 5 per hour)</h2>";
    
    $ipLimiter = new IPRateLimiter(2, 5);
    
    // Get initial state
    $ipRemaining = $ipLimiter->getRemainingRequests();
    displayResult("Initial minute limit should be 2", $ipRemaining['minute'] == 2, "Minute remaining: " . $ipRemaining['minute']);
    displayResult("Initial hour limit should be 5", $ipRemaining['hour'] == 5, "Hour remaining: " . $ipRemaining['hour']);
    
    // First request
    $ipCheck1 = $ipLimiter->isAllowed();
    displayResult("First IP request should be allowed", $ipCheck1['allowed']);
    
    // Second request
    $ipCheck2 = $ipLimiter->isAllowed();
    displayResult("Second IP request should be allowed", $ipCheck2['allowed']);
    
    // Third request (should hit minute limit)
    $ipCheck3 = $ipLimiter->isAllowed();
    displayResult("Third IP request should be blocked (minute limit)", !$ipCheck3['allowed'], $ipCheck3['reason'] ?? '');
    
    // Test IP wait time
    $ipWaitTime = $ipLimiter->getWaitTime();
    displayResult("IP wait time should be between 1-60 seconds", $ipWaitTime > 0 && $ipWaitTime <= 60, "Wait time: $ipWaitTime seconds");
    
    echo "</div>";

    // Test 3: Reset Time Calculation
    echo "<div class='test-section'>";
    echo "<h2>Test 3: Reset Time Calculation</h2>";
    
    $rateLimiter3 = new RateLimiter(1, 30); // 1 request per 30 seconds for easier testing
    
    // Make a request
    $rateLimiter3->isAllowed('reset_test');
    $resetTime = $rateLimiter3->getResetTime('reset_test');
    $currentTime = time();
    $timeDiff = $resetTime - $currentTime;
    
    displayResult("Reset time should be ~30 seconds in the future", $timeDiff >= 29 && $timeDiff <= 31, "Time until reset: $timeDiff seconds");
    
    // Make another request (should be blocked)
    $allowed = $rateLimiter3->isAllowed('reset_test');
    displayResult("Second request should be blocked", !$allowed);
    
    $waitTime = $rateLimiter3->getWaitTime('reset_test');
    displayResult("Wait time should match time until reset", abs($waitTime - $timeDiff) <= 1, "Wait time: $waitTime, Time diff: $timeDiff");
    
    echo "</div>";

    // Test 4: Multiple Actions
    echo "<div class='test-section'>";
    echo "<h2>Test 4: Multiple Actions (Different Limits)</h2>";
    
    $rateLimiter4 = new RateLimiter(3, 60);
    
    // Test different actions have separate limits
    $allowed_a1 = $rateLimiter4->isAllowed('action_a');
    $allowed_b1 = $rateLimiter4->isAllowed('action_b');
    $allowed_a2 = $rateLimiter4->isAllowed('action_a');
    
    $remaining_a = $rateLimiter4->getRemainingRequests('action_a');
    $remaining_b = $rateLimiter4->getRemainingRequests('action_b');
    
    displayResult("Action A should have 1 remaining", $remaining_a == 1, "Remaining: $remaining_a");
    displayResult("Action B should have 2 remaining", $remaining_b == 2, "Remaining: $remaining_b");
    
    echo "</div>";

    // Test 5: Suspicious Activity Detection
    echo "<div class='test-section'>";
    echo "<h2>Test 5: Suspicious Activity Detection</h2>";
    
    $ipLimiter5 = new IPRateLimiter(10, 50); // Higher limits for this test
    
    // Simulate rapid requests
    for ($i = 0; $i < 4; $i++) {
        $ipLimiter5->isAllowed();
    }
    
    $suspicious = $ipLimiter5->isSuspicious();
    displayResult("Should detect suspicious activity after 4 rapid requests", $suspicious);
    
    echo "</div>";

    // Test 6: Time Window Expiration
    echo "<div class='test-section'>";
    echo "<h2>Test 6: Time Window Expiration (Simulated)</h2>";
    
    echo "<div class='warning'>Note: This test simulates time passing. In production, you would need to wait for actual time to pass.</div>";
    
    $rateLimiter6 = new RateLimiter(2, 10); // 2 requests per 10 seconds for faster testing
    
    // Use up the limit
    $rateLimiter6->isAllowed('expiry_test');
    $rateLimiter6->isAllowed('expiry_test');
    
    $blocked = !$rateLimiter6->isAllowed('expiry_test');
    displayResult("Should be blocked after 2 requests", $blocked);
    
    // In a real scenario, we would wait 10 seconds
    // For testing, we'll just check the logic
    $waitTime = $rateLimiter6->getWaitTime('expiry_test');
    displayResult("Wait time should be <= 10 seconds", $waitTime <= 10, "Wait time: $waitTime seconds");
    
    echo "</div>";

    // Display current limits status
    echo "<div class='test-section'>";
    echo "<h2>Current Rate Limit Status</h2>";
    
    $statusLimiter = new RateLimiter(2, 60);
    $statusIPLimiter = new IPRateLimiter(2, 5);
    
    $sessionRemaining = $statusLimiter->getRemainingRequests('image_process');
    $ipRemaining = $statusIPLimiter->getRemainingRequests();
    
    echo "<pre>";
    echo "Session Rate Limit Status:\n";
    echo "  Remaining requests: " . $sessionRemaining . " / 2\n";
    echo "  Wait time: " . $statusLimiter->getWaitTime('image_process') . " seconds\n\n";
    
    echo "IP Rate Limit Status:\n";
    echo "  User IP: " . $statusIPLimiter->getUserIP() . "\n";
    echo "  Minute remaining: " . $ipRemaining['minute'] . " / 2\n";
    echo "  Hour remaining: " . $ipRemaining['hour'] . " / 5\n";
    echo "  Wait time: " . $statusIPLimiter->getWaitTime() . " seconds\n";
    echo "  Suspicious: " . ($statusIPLimiter->isSuspicious() ? 'Yes' : 'No') . "\n";
    echo "</pre>";
    
    echo "</div>";
    ?>
    
    <script>
        function clearAllLimits() {
            if (confirm('This will clear all rate limit data. Are you sure?')) {
                fetch('clear_rate_limits.php')
                    .then(response => response.json())
                    .then(data => {
                        alert(data.message || 'Rate limits cleared');
                        location.reload();
                    })
                    .catch(err => {
                        alert('Error clearing rate limits: ' + err.message);
                    });
            }
        }
    </script>
</body>
</html>