<?php
session_start();

// Include necessary files
require_once __DIR__ . '/includes/RateLimiter.php';
require_once __DIR__ . '/includes/IPRateLimiter.php';

// Create a test image in session if not exists
if (!isset($_SESSION['uploaded_image'])) {
    // Use the test image that already exists
    $_SESSION['uploaded_image'] = [
        'path' => __DIR__ . '/test_passport.jpg',
        'name' => 'test_passport.jpg',
        'type' => 'image/jpeg',
        'width' => 600,
        'height' => 600
    ];
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>Test Process Rate Limits</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 800px;
            margin: 40px auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #333;
            border-bottom: 2px solid #667eea;
            padding-bottom: 10px;
        }
        .test-area {
            margin: 20px 0;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 8px;
        }
        button {
            padding: 12px 24px;
            margin: 10px 5px;
            border: none;
            border-radius: 6px;
            font-size: 16px;
            cursor: pointer;
            transition: all 0.3s;
        }
        .test-btn {
            background: #667eea;
            color: white;
        }
        .test-btn:hover {
            background: #5a67d8;
        }
        .clear-btn {
            background: #dc3545;
            color: white;
        }
        .clear-btn:hover {
            background: #c82333;
        }
        .status {
            margin: 20px 0;
            padding: 15px;
            border-radius: 6px;
            background: #e8f4f8;
        }
        .log {
            margin: 20px 0;
            padding: 20px;
            background: #f5f5f5;
            border-radius: 6px;
            max-height: 400px;
            overflow-y: auto;
            font-family: monospace;
            font-size: 14px;
        }
        .log-entry {
            margin: 5px 0;
            padding: 8px;
            background: white;
            border-radius: 4px;
        }
        .success { color: #28a745; }
        .error { color: #dc3545; }
        .info { color: #17a2b8; }
        .rate-limit-modal {
            display: none;
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
            z-index: 1000;
            text-align: center;
        }
        .rate-limit-modal.active {
            display: block;
        }
        .overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.5);
            z-index: 999;
        }
        .overlay.active {
            display: block;
        }
        .countdown {
            font-size: 48px;
            color: #667eea;
            font-weight: bold;
            margin: 20px 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🧪 Test Process Rate Limits</h1>
        
        <div class="status">
            <h3>Current Status</h3>
            <div id="currentStatus">Loading...</div>
        </div>
        
        <div class="test-area">
            <h3>Test Controls</h3>
            <button class="test-btn" onclick="testSingleRequest()">📤 Make Single Request</button>
            <button class="test-btn" onclick="testMultipleRequests()">📤📤📤 Make 3 Rapid Requests</button>
            <button class="clear-btn" onclick="clearLimits()">🗑️ Clear All Limits</button>
            <button class="test-btn" onclick="updateStatus()">🔄 Refresh Status</button>
        </div>
        
        <div class="log">
            <h3>Test Log</h3>
            <div id="testLog"></div>
        </div>
    </div>
    
    <!-- Rate Limit Modal -->
    <div class="overlay" id="overlay"></div>
    <div class="rate-limit-modal" id="rateLimitModal">
        <h2>⏱️ Rate Limit Exceeded</h2>
        <p>Please wait before making another request</p>
        <div class="countdown" id="countdown">00:00</div>
        <button onclick="closeModal()">Close</button>
    </div>
    
    <script>
        let requestCount = 0;
        let countdownInterval = null;
        
        function log(message, type = 'info') {
            const logDiv = document.getElementById('testLog');
            const entry = document.createElement('div');
            entry.className = `log-entry ${type}`;
            const timestamp = new Date().toLocaleTimeString();
            entry.innerHTML = `[${timestamp}] ${message}`;
            logDiv.insertBefore(entry, logDiv.firstChild);
        }
        
        function updateStatus() {
            fetch('get_rate_limit_status.php')
                .then(response => response.json())
                .then(data => {
                    const statusDiv = document.getElementById('currentStatus');
                    statusDiv.innerHTML = `
                        <strong>Session Limits:</strong><br>
                        - Remaining: ${data.session.remaining} / ${data.session.max}<br>
                        - Wait time: ${data.session.waitTime} seconds<br><br>
                        <strong>IP Limits:</strong><br>
                        - Minute: ${data.ip.minute} / ${data.ip.maxMinute}<br>
                        - Hour: ${data.ip.hour} / ${data.ip.maxHour}<br>
                        - Wait time: ${data.ip.waitTime} seconds<br>
                        - Your IP: ${data.ip.address}
                    `;
                })
                .catch(err => {
                    log('Error fetching status: ' + err.message, 'error');
                });
        }
        
        function testSingleRequest() {
            requestCount++;
            log(`Making request #${requestCount}...`, 'info');
            
            const params = new URLSearchParams({
                width: 300,
                height: 300,
                cropX: 0,
                cropY: 0,
                cropWidth: 600,
                cropHeight: 600,
                brightness: 0,
                contrast: 0,
                background: 'none',
                quality: 85
            });
            
            fetch('process_image.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: params
            })
            .then(response => response.json())
            .then(data => {
                if (data.error) {
                    log(`Request #${requestCount} failed: ${data.error}`, 'error');
                    
                    // Check for rate limit error
                    if (data.error.includes('Rate limit exceeded')) {
                        const match = data.error.match(/Remaining time: (\d+) seconds/);
                        if (match) {
                            const seconds = parseInt(match[1]);
                            showRateLimitModal(seconds);
                        }
                    }
                } else {
                    log(`Request #${requestCount} succeeded! File size: ${data.size} KB`, 'success');
                }
                updateStatus();
            })
            .catch(err => {
                log(`Request #${requestCount} error: ${err.message}`, 'error');
                updateStatus();
            });
        }
        
        function testMultipleRequests() {
            log('Starting rapid request test...', 'info');
            
            // Make 3 requests with small delays
            for (let i = 0; i < 3; i++) {
                setTimeout(() => {
                    testSingleRequest();
                }, i * 100); // 100ms delay between requests
            }
        }
        
        function clearLimits() {
            if (confirm('Clear all rate limit data?')) {
                fetch('clear_rate_limits.php')
                    .then(response => response.json())
                    .then(data => {
                        log(data.message, data.success ? 'success' : 'error');
                        requestCount = 0;
                        updateStatus();
                    })
                    .catch(err => {
                        log('Error clearing limits: ' + err.message, 'error');
                    });
            }
        }
        
        function showRateLimitModal(seconds) {
            document.getElementById('overlay').classList.add('active');
            document.getElementById('rateLimitModal').classList.add('active');
            
            if (countdownInterval) {
                clearInterval(countdownInterval);
            }
            
            let remaining = seconds;
            updateCountdown(remaining);
            
            countdownInterval = setInterval(() => {
                remaining--;
                updateCountdown(remaining);
                
                if (remaining <= 0) {
                    clearInterval(countdownInterval);
                    closeModal();
                    log('Rate limit expired - you can make requests again', 'success');
                    updateStatus();
                }
            }, 1000);
        }
        
        function updateCountdown(seconds) {
            const minutes = Math.floor(seconds / 60);
            const secs = seconds % 60;
            document.getElementById('countdown').textContent = 
                `${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
        }
        
        function closeModal() {
            document.getElementById('overlay').classList.remove('active');
            document.getElementById('rateLimitModal').classList.remove('active');
            if (countdownInterval) {
                clearInterval(countdownInterval);
            }
        }
        
        // Initial status update
        updateStatus();
        log('Test page loaded - ready to test rate limits', 'info');
    </script>
</body>
</html>