<?php
session_start();

// For testing, create a simple test session if not exists
if (!isset($_SESSION['uploaded_image'])) {
    $_SESSION['uploaded_image'] = [
        'path' => 'test_passport.jpg',
        'name' => 'test_passport.jpg',
        'type' => 'image/jpeg',
        'width' => 600,
        'height' => 600
    ];
}

$imageInfo = $_SESSION['uploaded_image'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Crop Test - Photo Resizer</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: Arial, sans-serif;
            background: #f0f0f0;
            padding: 20px;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            padding: 20px;
            border-radius: 10px;
        }
        
        .workspace {
            display: grid;
            grid-template-columns: 1fr 300px;
            gap: 20px;
            margin-top: 20px;
        }
        
        .image-area {
            background: #f8f8f8;
            padding: 20px;
            border-radius: 8px;
            text-align: center;
        }
        
        .crop-container {
            position: relative;
            display: inline-block;
            background: #ddd;
            user-select: none;
        }
        
        #sourceImage {
            display: block;
            max-width: 100%;
            height: auto;
        }
        
        .crop-box {
            position: absolute;
            border: 2px solid #007bff;
            background: rgba(0, 123, 255, 0.1);
            cursor: move;
        }
        
        .crop-handle {
            position: absolute;
            width: 12px;
            height: 12px;
            background: #007bff;
            border: 2px solid white;
            box-shadow: 0 0 3px rgba(0,0,0,0.5);
        }
        
        .crop-handle.tl { top: -6px; left: -6px; cursor: nw-resize; }
        .crop-handle.tr { top: -6px; right: -6px; cursor: ne-resize; }
        .crop-handle.bl { bottom: -6px; left: -6px; cursor: sw-resize; }
        .crop-handle.br { bottom: -6px; right: -6px; cursor: se-resize; }
        .crop-handle.tm { top: -6px; left: 50%; transform: translateX(-50%); cursor: n-resize; }
        .crop-handle.bm { bottom: -6px; left: 50%; transform: translateX(-50%); cursor: s-resize; }
        .crop-handle.ml { left: -6px; top: 50%; transform: translateY(-50%); cursor: w-resize; }
        .crop-handle.mr { right: -6px; top: 50%; transform: translateY(-50%); cursor: e-resize; }
        
        .controls {
            background: #f8f8f8;
            padding: 20px;
            border-radius: 8px;
        }
        
        .control-group {
            margin-bottom: 15px;
        }
        
        .control-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        
        .control-group input, .control-group select {
            width: 100%;
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        
        .preview {
            margin-top: 20px;
            text-align: center;
        }
        
        #previewCanvas {
            border: 1px solid #ddd;
            max-width: 100%;
        }
        
        .debug {
            margin-top: 20px;
            padding: 10px;
            background: #f0f0f0;
            border-radius: 4px;
            font-family: monospace;
            font-size: 12px;
        }
        
        button {
            background: #007bff;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 4px;
            cursor: pointer;
            width: 100%;
            margin-top: 10px;
        }
        
        button:hover {
            background: #0056b3;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Crop Functionality Test</h1>
        
        <div class="workspace">
            <div class="image-area">
                <h3>Original Image</h3>
                <div class="crop-container" id="cropContainer">
                    <img id="sourceImage" src="<?php echo htmlspecialchars($imageInfo['path']); ?>" alt="Source">
                    <div class="crop-box" id="cropBox">
                        <div class="crop-handle tl"></div>
                        <div class="crop-handle tr"></div>
                        <div class="crop-handle bl"></div>
                        <div class="crop-handle br"></div>
                        <div class="crop-handle tm"></div>
                        <div class="crop-handle bm"></div>
                        <div class="crop-handle ml"></div>
                        <div class="crop-handle mr"></div>
                    </div>
                </div>
                
                <div class="debug" id="debug">
                    Debug info will appear here
                </div>
            </div>
            
            <div class="controls">
                <h3>Controls</h3>
                
                <div class="control-group">
                    <label>Preset Size:</label>
                    <select id="presetSize">
                        <option value="">Free Crop</option>
                        <option value="413x531">3.5x4.5cm (413x531px)</option>
                        <option value="240x320">240x320px</option>
                        <option value="200x230">200x230px</option>
                        <option value="350x350">350x350px (Square)</option>
                    </select>
                </div>
                
                <div class="control-group">
                    <label>Crop X: <span id="cropXValue">0</span></label>
                    <input type="range" id="cropX" min="0" max="600" value="0">
                </div>
                
                <div class="control-group">
                    <label>Crop Y: <span id="cropYValue">0</span></label>
                    <input type="range" id="cropY" min="0" max="600" value="0">
                </div>
                
                <div class="control-group">
                    <label>Crop Width: <span id="cropWidthValue">200</span></label>
                    <input type="range" id="cropWidth" min="50" max="600" value="200">
                </div>
                
                <div class="control-group">
                    <label>Crop Height: <span id="cropHeightValue">200</span></label>
                    <input type="range" id="cropHeight" min="50" max="600" value="200">
                </div>
                
                <button onclick="resetCrop()">Reset Crop</button>
                <button onclick="updatePreview()">Update Preview</button>
                
                <div class="preview">
                    <h3>Preview</h3>
                    <canvas id="previewCanvas"></canvas>
                    <p>Size: <span id="previewSize">-</span></p>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Global variables
        let cropData = {
            x: 0,
            y: 0,
            width: 200,
            height: 200
        };
        
        let dragState = {
            active: false,
            element: null,
            startX: 0,
            startY: 0,
            startCropX: 0,
            startCropY: 0,
            startCropWidth: 0,
            startCropHeight: 0
        };
        
        let imageData = {
            width: 0,
            height: 0,
            displayWidth: 0,
            displayHeight: 0,
            scale: 1
        };
        
        // Initialize when page loads
        window.addEventListener('load', function() {
            const img = document.getElementById('sourceImage');
            img.onload = function() {
                initializeImage();
                initializeCrop();
                setupEventHandlers();
                updateCropBox();
                updatePreview();
            };
            
            // If image is already loaded
            if (img.complete) {
                img.onload();
            }
        });
        
        function initializeImage() {
            const img = document.getElementById('sourceImage');
            const container = document.getElementById('cropContainer');
            
            imageData.width = img.naturalWidth;
            imageData.height = img.naturalHeight;
            imageData.displayWidth = img.offsetWidth;
            imageData.displayHeight = img.offsetHeight;
            imageData.scale = img.naturalWidth / img.offsetWidth;
            
            // Set container size to match image
            container.style.width = img.offsetWidth + 'px';
            container.style.height = img.offsetHeight + 'px';
            
            updateDebug();
        }
        
        function initializeCrop() {
            const img = document.getElementById('sourceImage');
            
            // Center crop box on image
            cropData.width = Math.min(200, img.offsetWidth * 0.8);
            cropData.height = Math.min(200, img.offsetHeight * 0.8);
            cropData.x = (img.offsetWidth - cropData.width) / 2;
            cropData.y = (img.offsetHeight - cropData.height) / 2;
            
            // Update sliders
            document.getElementById('cropX').max = img.offsetWidth - cropData.width;
            document.getElementById('cropY').max = img.offsetHeight - cropData.height;
            document.getElementById('cropWidth').max = img.offsetWidth;
            document.getElementById('cropHeight').max = img.offsetHeight;
            
            updateSliders();
        }
        
        function setupEventHandlers() {
            const cropBox = document.getElementById('cropBox');
            const handles = cropBox.querySelectorAll('.crop-handle');
            
            // Crop box drag
            cropBox.addEventListener('mousedown', function(e) {
                if (e.target === cropBox) {
                    startDrag(e, 'move');
                }
            });
            
            // Handle drag
            handles.forEach(handle => {
                handle.addEventListener('mousedown', function(e) {
                    const classes = this.className.split(' ');
                    startDrag(e, classes[1]); // Get handle type (tl, tr, etc.)
                    e.stopPropagation();
                });
            });
            
            // Global mouse events
            document.addEventListener('mousemove', handleDrag);
            document.addEventListener('mouseup', stopDrag);
            
            // Slider events
            ['cropX', 'cropY', 'cropWidth', 'cropHeight'].forEach(id => {
                document.getElementById(id).addEventListener('input', function() {
                    cropData[id.replace('crop', '').toLowerCase()] = parseInt(this.value);
                    updateCropBox();
                    updateSliderLabels();
                    updateDebug();
                });
            });
            
            // Preset size change
            document.getElementById('presetSize').addEventListener('change', function() {
                if (this.value) {
                    const [width, height] = this.value.split('x').map(Number);
                    applyPresetSize(width, height);
                }
            });
        }
        
        function startDrag(e, type) {
            dragState.active = true;
            dragState.element = type;
            dragState.startX = e.clientX;
            dragState.startY = e.clientY;
            dragState.startCropX = cropData.x;
            dragState.startCropY = cropData.y;
            dragState.startCropWidth = cropData.width;
            dragState.startCropHeight = cropData.height;
            
            e.preventDefault();
            document.body.style.cursor = e.target.style.cursor || 'move';
        }
        
        function handleDrag(e) {
            if (!dragState.active) return;
            
            const dx = e.clientX - dragState.startX;
            const dy = e.clientY - dragState.startY;
            const img = document.getElementById('sourceImage');
            
            switch(dragState.element) {
                case 'move':
                    cropData.x = Math.max(0, Math.min(img.offsetWidth - cropData.width, dragState.startCropX + dx));
                    cropData.y = Math.max(0, Math.min(img.offsetHeight - cropData.height, dragState.startCropY + dy));
                    break;
                
                case 'tl': // Top-left
                    cropData.x = Math.max(0, Math.min(dragState.startCropX + dragState.startCropWidth - 50, dragState.startCropX + dx));
                    cropData.y = Math.max(0, Math.min(dragState.startCropY + dragState.startCropHeight - 50, dragState.startCropY + dy));
                    cropData.width = dragState.startCropWidth - (cropData.x - dragState.startCropX);
                    cropData.height = dragState.startCropHeight - (cropData.y - dragState.startCropY);
                    break;
                
                case 'tr': // Top-right
                    cropData.y = Math.max(0, Math.min(dragState.startCropY + dragState.startCropHeight - 50, dragState.startCropY + dy));
                    cropData.width = Math.max(50, Math.min(img.offsetWidth - cropData.x, dragState.startCropWidth + dx));
                    cropData.height = dragState.startCropHeight - (cropData.y - dragState.startCropY);
                    break;
                
                case 'bl': // Bottom-left
                    cropData.x = Math.max(0, Math.min(dragState.startCropX + dragState.startCropWidth - 50, dragState.startCropX + dx));
                    cropData.width = dragState.startCropWidth - (cropData.x - dragState.startCropX);
                    cropData.height = Math.max(50, Math.min(img.offsetHeight - cropData.y, dragState.startCropHeight + dy));
                    break;
                
                case 'br': // Bottom-right
                    cropData.width = Math.max(50, Math.min(img.offsetWidth - cropData.x, dragState.startCropWidth + dx));
                    cropData.height = Math.max(50, Math.min(img.offsetHeight - cropData.y, dragState.startCropHeight + dy));
                    break;
                
                case 'tm': // Top-middle
                    cropData.y = Math.max(0, Math.min(dragState.startCropY + dragState.startCropHeight - 50, dragState.startCropY + dy));
                    cropData.height = dragState.startCropHeight - (cropData.y - dragState.startCropY);
                    break;
                
                case 'bm': // Bottom-middle
                    cropData.height = Math.max(50, Math.min(img.offsetHeight - cropData.y, dragState.startCropHeight + dy));
                    break;
                
                case 'ml': // Middle-left
                    cropData.x = Math.max(0, Math.min(dragState.startCropX + dragState.startCropWidth - 50, dragState.startCropX + dx));
                    cropData.width = dragState.startCropWidth - (cropData.x - dragState.startCropX);
                    break;
                
                case 'mr': // Middle-right
                    cropData.width = Math.max(50, Math.min(img.offsetWidth - cropData.x, dragState.startCropWidth + dx));
                    break;
            }
            
            updateCropBox();
            updateSliders();
            updateDebug();
        }
        
        function stopDrag() {
            if (dragState.active) {
                dragState.active = false;
                document.body.style.cursor = '';
                updatePreview();
            }
        }
        
        function updateCropBox() {
            const cropBox = document.getElementById('cropBox');
            cropBox.style.left = cropData.x + 'px';
            cropBox.style.top = cropData.y + 'px';
            cropBox.style.width = cropData.width + 'px';
            cropBox.style.height = cropData.height + 'px';
        }
        
        function updateSliders() {
            document.getElementById('cropX').value = cropData.x;
            document.getElementById('cropY').value = cropData.y;
            document.getElementById('cropWidth').value = cropData.width;
            document.getElementById('cropHeight').value = cropData.height;
            
            // Update max values
            const img = document.getElementById('sourceImage');
            document.getElementById('cropX').max = img.offsetWidth - cropData.width;
            document.getElementById('cropY').max = img.offsetHeight - cropData.height;
            
            updateSliderLabels();
        }
        
        function updateSliderLabels() {
            document.getElementById('cropXValue').textContent = Math.round(cropData.x);
            document.getElementById('cropYValue').textContent = Math.round(cropData.y);
            document.getElementById('cropWidthValue').textContent = Math.round(cropData.width);
            document.getElementById('cropHeightValue').textContent = Math.round(cropData.height);
        }
        
        function updateDebug() {
            const debug = document.getElementById('debug');
            debug.innerHTML = `
                Image: ${imageData.width}x${imageData.height} (natural)<br>
                Display: ${imageData.displayWidth}x${imageData.displayHeight}<br>
                Scale: ${imageData.scale.toFixed(2)}<br>
                Crop: [${Math.round(cropData.x)}, ${Math.round(cropData.y)}, ${Math.round(cropData.width)}, ${Math.round(cropData.height)}]<br>
                Real crop: [${Math.round(cropData.x * imageData.scale)}, ${Math.round(cropData.y * imageData.scale)}, ${Math.round(cropData.width * imageData.scale)}, ${Math.round(cropData.height * imageData.scale)}]
            `;
        }
        
        function updatePreview() {
            const canvas = document.getElementById('previewCanvas');
            const ctx = canvas.getContext('2d');
            const img = document.getElementById('sourceImage');
            
            // Set canvas size based on crop
            canvas.width = Math.min(cropData.width, 250);
            canvas.height = Math.min(cropData.height, 250);
            
            // Clear canvas
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            
            // Draw cropped portion
            ctx.drawImage(
                img,
                cropData.x * imageData.scale,
                cropData.y * imageData.scale,
                cropData.width * imageData.scale,
                cropData.height * imageData.scale,
                0, 0,
                canvas.width,
                canvas.height
            );
            
            // Update size display
            document.getElementById('previewSize').textContent = 
                `${Math.round(cropData.width * imageData.scale)}x${Math.round(cropData.height * imageData.scale)}px`;
        }
        
        function applyPresetSize(width, height) {
            const img = document.getElementById('sourceImage');
            const aspectRatio = width / height;
            
            // Try to fit the preset size within current crop area
            let newWidth = cropData.width;
            let newHeight = newWidth / aspectRatio;
            
            // If height is too large, scale based on height
            if (newHeight > img.offsetHeight - cropData.y) {
                newHeight = img.offsetHeight - cropData.y;
                newWidth = newHeight * aspectRatio;
            }
            
            // If width is too large, scale based on width
            if (newWidth > img.offsetWidth - cropData.x) {
                newWidth = img.offsetWidth - cropData.x;
                newHeight = newWidth / aspectRatio;
            }
            
            // Apply new dimensions
            cropData.width = Math.max(50, newWidth);
            cropData.height = Math.max(50, newHeight);
            
            updateCropBox();
            updateSliders();
            updateDebug();
            updatePreview();
        }
        
        function resetCrop() {
            initializeCrop();
            updateCropBox();
            updatePreview();
            updateDebug();
            document.getElementById('presetSize').value = '';
        }
    </script>
</body>
</html>