#!/usr/bin/env php
<?php
// Command line test script for rate limiters

require_once __DIR__ . '/includes/RateLimiter.php';
require_once __DIR__ . '/includes/IPRateLimiter.php';

echo "\n=== RATE LIMITER TEST RESULTS ===\n\n";

// Test 1: Session Rate Limiter
echo "1. SESSION RATE LIMITER TEST (2 requests per 60 seconds)\n";
echo str_repeat('-', 50) . "\n";

$rateLimiter = new RateLimiter(2, 60);
$action = 'cli_test_' . uniqid();

echo "Initial state:\n";
echo "  Remaining requests: " . $rateLimiter->getRemainingRequests($action) . "\n";
echo "  Wait time: " . $rateLimiter->getWaitTime($action) . " seconds\n\n";

// Make requests
for ($i = 1; $i <= 4; $i++) {
    $allowed = $rateLimiter->isAllowed($action);
    echo "Request #$i: " . ($allowed ? "✅ ALLOWED" : "❌ BLOCKED") . "\n";
    if (!$allowed) {
        echo "  Wait time: " . $rateLimiter->getWaitTime($action) . " seconds\n";
    }
    echo "  Remaining: " . $rateLimiter->getRemainingRequests($action) . "\n";
}

echo "\n2. IP RATE LIMITER TEST (2 per minute, 5 per hour)\n";
echo str_repeat('-', 50) . "\n";

$ipLimiter = new IPRateLimiter(2, 5);

echo "Initial state:\n";
$remaining = $ipLimiter->getRemainingRequests();
echo "  Minute remaining: " . $remaining['minute'] . "\n";
echo "  Hour remaining: " . $remaining['hour'] . "\n";
echo "  User IP: " . $ipLimiter->getUserIP() . "\n\n";

// Make requests
for ($i = 1; $i <= 3; $i++) {
    $check = $ipLimiter->isAllowed();
    echo "Request #$i: " . ($check['allowed'] ? "✅ ALLOWED" : "❌ BLOCKED") . "\n";
    if (!$check['allowed']) {
        echo "  Reason: " . $check['reason'] . "\n";
        echo "  Wait time: " . $ipLimiter->getWaitTime() . " seconds\n";
    }
    $remaining = $ipLimiter->getRemainingRequests();
    echo "  Minute remaining: " . $remaining['minute'] . "\n";
    echo "  Hour remaining: " . $remaining['hour'] . "\n";
}

echo "\n3. RESET TIME TEST\n";
echo str_repeat('-', 50) . "\n";

$testLimiter = new RateLimiter(1, 30); // 1 request per 30 seconds
$testAction = 'reset_test_' . uniqid();

// Make one request
$testLimiter->isAllowed($testAction);
$resetTime = $testLimiter->getResetTime($testAction);
$waitTime = $testLimiter->getWaitTime($testAction);

echo "After 1 request (limit is 1):\n";
echo "  Current time: " . time() . "\n";
echo "  Reset time: " . $resetTime . "\n";
echo "  Time until reset: " . ($resetTime - time()) . " seconds\n";
echo "  Wait time method: " . $waitTime . " seconds\n";

// Try another request
$blocked = !$testLimiter->isAllowed($testAction);
echo "\nSecond request: " . ($blocked ? "❌ BLOCKED (as expected)" : "✅ ALLOWED (unexpected!)") . "\n";
echo "  New wait time: " . $testLimiter->getWaitTime($testAction) . " seconds\n";

echo "\n4. SUSPICIOUS ACTIVITY TEST\n";
echo str_repeat('-', 50) . "\n";

$suspiciousLimiter = new IPRateLimiter(10, 50); // Higher limits
echo "Making 4 rapid requests...\n";

for ($i = 1; $i <= 4; $i++) {
    $suspiciousLimiter->isAllowed();
}

$isSuspicious = $suspiciousLimiter->isSuspicious();
echo "Suspicious activity detected: " . ($isSuspicious ? "✅ YES" : "❌ NO") . "\n";

echo "\n=== TEST COMPLETE ===\n";
?>