<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/error.log');

// Set memory limit for this process
ini_set('memory_limit', '128M');

// Set execution time limit
set_time_limit(30);

session_start();
header('Content-Type: application/json');

// Include optimization classes
require_once __DIR__ . '/includes/RateLimiter.php';
require_once __DIR__ . '/includes/IPRateLimiter.php';
require_once __DIR__ . '/includes/ProcessingQueue.php';
require_once __DIR__ . '/includes/ServerMonitor.php';

try {
    // Initialize rate limiter (2 requests per minute)
    $rateLimiter = new RateLimiter(2, 60);
    
    // Check session rate limit
    if (!$rateLimiter->isAllowed('image_process')) {
        $waitTime = $rateLimiter->getWaitTime('image_process');
        throw new Exception('You\'ve reached your processing limit (2 per minute). Please wait ' . 
            $waitTime . ' seconds before trying again.');
    }
    
    // Check IP rate limit (2 per minute, 5 per hour)
    $ipLimiter = new IPRateLimiter(2, 5);
    $ipCheck = $ipLimiter->isAllowed();
    
    if (!$ipCheck['allowed']) {
        $waitTime = $ipLimiter->getWaitTime();
        throw new Exception($ipCheck['reason'] . ' Remaining time: ' . $waitTime . ' seconds');
    }
    
    // Check if IP is suspicious
    if ($ipLimiter->isSuspicious()) {
        throw new Exception('Suspicious activity detected. Please complete the CAPTCHA on the upload page.');
    }
    
    // Check server resources
    $monitor = new ServerMonitor();
    if (!$monitor->checkResources()) {
        throw new Exception('Server is currently under high load. Please try again in a few moments.');
    }
    
    // Initialize processing queue (max 25 concurrent processes)
    $queue = new ProcessingQueue(25, 30);
    
    // Check if image was uploaded
    if (!isset($_SESSION['uploaded_image'])) {
        throw new Exception('No image found in session. Please upload an image first.');
    }
    
    if (!isset($_SESSION['uploaded_image']['path'])) {
        throw new Exception('Invalid session data - path missing');
    }
    
    if (!file_exists($_SESSION['uploaded_image']['path'])) {
        throw new Exception('Image file not found: ' . $_SESSION['uploaded_image']['path']);
    }

    // Check if it's a POST request
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Invalid request method: ' . $_SERVER['REQUEST_METHOD']);
    }

    // Get parameters with defaults
    $width = isset($_POST['width']) ? intval($_POST['width']) : 0;
    $height = isset($_POST['height']) ? intval($_POST['height']) : 0;
    $cropX = isset($_POST['cropX']) ? intval($_POST['cropX']) : 0;
    $cropY = isset($_POST['cropY']) ? intval($_POST['cropY']) : 0;
    $cropWidth = isset($_POST['cropWidth']) ? intval($_POST['cropWidth']) : 0;
    $cropHeight = isset($_POST['cropHeight']) ? intval($_POST['cropHeight']) : 0;
    $brightness = isset($_POST['brightness']) ? intval($_POST['brightness']) : 0;
    $contrast = isset($_POST['contrast']) ? intval($_POST['contrast']) : 0;
    $background = isset($_POST['background']) ? $_POST['background'] : 'none';
    $quality = isset($_POST['quality']) ? intval($_POST['quality']) : 85;

    // Validate parameters
    if ($width <= 0 || $height <= 0) {
        throw new Exception("Invalid target dimensions: {$width}x{$height}");
    }
    
    if ($cropWidth <= 0 || $cropHeight <= 0) {
        throw new Exception("Invalid crop dimensions: {$cropWidth}x{$cropHeight}");
    }

    // Process the image
    $imagePath = $_SESSION['uploaded_image']['path'];
    
    // Get image info using getimagesize instead of exif_imagetype
    $imageInfo = @getimagesize($imagePath);
    if ($imageInfo === false) {
        throw new Exception('Cannot determine image type');
    }
    
    $imageType = $imageInfo[2];
    
    // Create image resource
    switch ($imageType) {
        case IMAGETYPE_JPEG:
            $source = @imagecreatefromjpeg($imagePath);
            break;
        case IMAGETYPE_PNG:
            $source = @imagecreatefrompng($imagePath);
            break;
        case IMAGETYPE_GIF:
            $source = @imagecreatefromgif($imagePath);
            break;
        default:
            throw new Exception('Unsupported image type: ' . $imageType);
    }
    
    if (!$source) {
        throw new Exception('Failed to create image resource from file');
    }
    
    // Get original dimensions
    $origWidth = imagesx($source);
    $origHeight = imagesy($source);
    
    // Validate and adjust crop coordinates
    $cropX = max(0, min($cropX, $origWidth - 1));
    $cropY = max(0, min($cropY, $origHeight - 1));
    $cropWidth = min($cropWidth, $origWidth - $cropX);
    $cropHeight = min($cropHeight, $origHeight - $cropY);
    
    if ($cropWidth <= 0 || $cropHeight <= 0) {
        throw new Exception("Invalid crop area after adjustment: {$cropWidth}x{$cropHeight} at ({$cropX},{$cropY})");
    }
    
    // Create cropped image
    $cropped = @imagecreatetruecolor($cropWidth, $cropHeight);
    if (!$cropped) {
        throw new Exception('Failed to create cropped image');
    }
    
    // Copy the cropped portion
    $copyResult = @imagecopyresampled($cropped, $source, 0, 0, $cropX, $cropY, $cropWidth, $cropHeight, $cropWidth, $cropHeight);
    if (!$copyResult) {
        throw new Exception('Failed to copy cropped area');
    }
    
    // Create final resized image
    $resized = @imagecreatetruecolor($width, $height);
    if (!$resized) {
        throw new Exception('Failed to create resized image');
    }
    
    // Apply background if needed
    if ($background !== 'none') {
        // Set background color
        if ($background === 'white') {
            $bgColor = imagecolorallocate($resized, 255, 255, 255);
        } else {
            $bgColor = imagecolorallocate($resized, 240, 240, 240); // Light gray
        }
        
        // Fill the entire canvas with background color
        imagefilledrectangle($resized, 0, 0, $width - 1, $height - 1, $bgColor);
    }
    
    // Copy cropped image to resized canvas
    // This will overlay the image on top of the background (if set)
    $resizeResult = @imagecopyresampled($resized, $cropped, 0, 0, 0, 0, $width, $height, $cropWidth, $cropHeight);
    if (!$resizeResult) {
        throw new Exception('Failed to resize image');
    }
    
    // Apply brightness/contrast
    if ($brightness !== 0) {
        imagefilter($resized, IMG_FILTER_BRIGHTNESS, $brightness);
    }
    if ($contrast !== 0) {
        imagefilter($resized, IMG_FILTER_CONTRAST, -$contrast);
    }
    
    // Create output buffer
    ob_start();
    imagejpeg($resized, null, $quality);
    $imageData = ob_get_clean();
    
    // Calculate file size
    $fileSize = strlen($imageData);
    
    // Clean up
    imagedestroy($source);
    imagedestroy($cropped);
    imagedestroy($resized);
    
    // Return success response
    echo json_encode([
        'success' => true,
        'data' => 'data:image/jpeg;base64,' . base64_encode($imageData),
        'size' => round($fileSize / 1024, 1)
    ]);
    
} catch (Exception $e) {
    // Log the error
    error_log('Process Image Error: ' . $e->getMessage() . ' in ' . $e->getFile() . ':' . $e->getLine());
    
    // Return error response
    echo json_encode([
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'post' => $_POST,
            'session' => isset($_SESSION['uploaded_image']) ? 'set' : 'not set'
        ]
    ]);
}
?>