<?php
session_start();

// Check if image was uploaded
if (!isset($_SESSION['uploaded_image']) || !file_exists($_SESSION['uploaded_image']['path'])) {
    header('Location: index.php');
    exit;
}

$imageInfo = $_SESSION['uploaded_image'];

// Preset sizes
$presetSizes = [
    '3.5x4.5cm (413x531px)' => ['width' => 413, 'height' => 531],
    '240x320px' => ['width' => 240, 'height' => 320],
    '200x230px' => ['width' => 200, 'height' => 230],
    '350x350px' => ['width' => 350, 'height' => 350],
    '600x600px' => ['width' => 600, 'height' => 600],
    '800x800px' => ['width' => 800, 'height' => 800],
    'Custom' => ['width' => 0, 'height' => 0]
];

// Handle AJAX requests
if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
    header('Content-Type: application/json');
    
    if ($_POST['action'] === 'preview') {
        try {
            $width = intval($_POST['width']);
            $height = intval($_POST['height']);
            $cropX = intval($_POST['cropX'] ?? 0);
            $cropY = intval($_POST['cropY'] ?? 0);
            $cropWidth = intval($_POST['cropWidth'] ?? $imageInfo['width']);
            $cropHeight = intval($_POST['cropHeight'] ?? $imageInfo['height']);
            $brightness = intval($_POST['brightness'] ?? 0);
            $contrast = intval($_POST['contrast'] ?? 0);
            $background = $_POST['background'] ?? 'white';
            $quality = intval($_POST['quality'] ?? 85);
            
            // Process image
            $processedImage = processImage(
                $imageInfo['path'],
                $width, $height,
                $cropX, $cropY, $cropWidth, $cropHeight,
                $brightness, $contrast, $background, $quality
            );
            
            // Get file size
            ob_start();
            imagejpeg($processedImage, null, $quality);
            $imageData = ob_get_contents();
            ob_end_clean();
            
            $fileSize = strlen($imageData);
            $base64 = base64_encode($imageData);
            
            imagedestroy($processedImage);
            
            echo json_encode([
                'success' => true,
                'preview' => 'data:image/jpeg;base64,' . $base64,
                'fileSize' => $fileSize,
                'fileSizeKB' => round($fileSize / 1024, 1),
                'warning' => ($fileSize < 20480 || $fileSize > 102400) ? 
                    'File size should be between 20KB and 100KB for optimal results' : ''
            ]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
        exit;
    }
    
    if ($_POST['action'] === 'download') {
        try {
            $width = intval($_POST['width']);
            $height = intval($_POST['height']);
            $cropX = intval($_POST['cropX'] ?? 0);
            $cropY = intval($_POST['cropY'] ?? 0);
            $cropWidth = intval($_POST['cropWidth'] ?? $imageInfo['width']);
            $cropHeight = intval($_POST['cropHeight'] ?? $imageInfo['height']);
            $brightness = intval($_POST['brightness'] ?? 0);
            $contrast = intval($_POST['contrast'] ?? 0);
            $background = $_POST['background'] ?? 'white';
            $quality = intval($_POST['quality'] ?? 85);
            $dpi = intval($_POST['dpi'] ?? 200);
            
            // Process image
            $processedImage = processImage(
                $imageInfo['path'],
                $width, $height,
                $cropX, $cropY, $cropWidth, $cropHeight,
                $brightness, $contrast, $background, $quality
            );
            
            // Create temporary file for download
            $tempFile = tempnam(sys_get_temp_dir(), 'resized_');
            imagejpeg($processedImage, $tempFile, $quality);
            imagedestroy($processedImage);
            
            // Store download info in session
            $_SESSION['download_file'] = [
                'path' => $tempFile,
                'name' => 'photo_' . $width . 'x' . $height . '_' . $dpi . 'dpi.jpg',
                'dpi' => $dpi
            ];
            
            echo json_encode(['success' => true, 'downloadUrl' => 'download.php']);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
        exit;
    }
}

function processImage($imagePath, $width, $height, $cropX, $cropY, $cropWidth, $cropHeight, $brightness, $contrast, $background, $quality) {
    // Load source image
    $sourceImage = imagecreatefromstring(file_get_contents($imagePath));
    if (!$sourceImage) {
        throw new Exception('Failed to load image');
    }
    
    // Create cropped image
    $croppedImage = imagecreatetruecolor($cropWidth, $cropHeight);
    
    // Enable alpha blending for better quality
    imagealphablending($croppedImage, true);
    imagesavealpha($croppedImage, true);
    
    imagecopy($croppedImage, $sourceImage, 0, 0, $cropX, $cropY, $cropWidth, $cropHeight);
    
    // Create final image with desired dimensions
    $finalImage = imagecreatetruecolor($width, $height);
    imagealphablending($finalImage, true);
    
    // Handle background options
    $bgColor = null;
    $replaceBackground = false;
    
    switch ($background) {
        case 'white':
            $bgColor = imagecolorallocate($finalImage, 255, 255, 255);
            $replaceBackground = true;
            break;
        case 'light':
            $bgColor = imagecolorallocate($finalImage, 245, 245, 245);
            $replaceBackground = true;
            break;
        case 'blue':
            $bgColor = imagecolorallocate($finalImage, 220, 235, 250);
            $replaceBackground = true;
            break;
        case 'keep':
        default:
            $bgColor = imagecolorallocate($finalImage, 255, 255, 255);
            $replaceBackground = false;
            break;
    }
    
    // Fill with background color
    imagefill($finalImage, 0, 0, $bgColor);
    
    // If we need to replace background, process the image
    if ($replaceBackground) {
        $croppedImage = replaceBackground($croppedImage, $cropWidth, $cropHeight, $background);
    }
    
    // Calculate position to fit the cropped area into final dimensions
    $scale = min($width / $cropWidth, $height / $cropHeight);
    $newWidth = $cropWidth * $scale;
    $newHeight = $cropHeight * $scale;
    $x = ($width - $newWidth) / 2;
    $y = ($height - $newHeight) / 2;
    
    // Resize and place image
    imagecopyresampled($finalImage, $croppedImage, $x, $y, 0, 0, $newWidth, $newHeight, $cropWidth, $cropHeight);
    
    // Apply brightness and contrast
    if ($brightness != 0 || $contrast != 0) {
        imagefilter($finalImage, IMG_FILTER_BRIGHTNESS, $brightness);
        if ($contrast != 0) {
            imagefilter($finalImage, IMG_FILTER_CONTRAST, -$contrast);
        }
    }
    
    imagedestroy($sourceImage);
    imagedestroy($croppedImage);
    
    return $finalImage;
}

function replaceBackground($image, $width, $height, $newBgType) {
    // This function replaces the background for passport photos
    // It uses edge detection and color sampling to identify the background
    
    // First, detect the background color by sampling edges
    $edgePixels = [];
    $samplePoints = 10;
    
    // Sample top and bottom edges
    for ($i = 0; $i < $samplePoints; $i++) {
        $x = intval($width * $i / $samplePoints);
        // Top edge
        $rgb = imagecolorat($image, $x, 0);
        $edgePixels[] = imagecolorsforindex($image, $rgb);
        // Bottom edge
        $rgb = imagecolorat($image, $x, $height - 1);
        $edgePixels[] = imagecolorsforindex($image, $rgb);
    }
    
    // Sample left and right edges
    for ($i = 0; $i < $samplePoints; $i++) {
        $y = intval($height * $i / $samplePoints);
        // Left edge
        $rgb = imagecolorat($image, 0, $y);
        $edgePixels[] = imagecolorsforindex($image, $rgb);
        // Right edge
        $rgb = imagecolorat($image, $width - 1, $y);
        $edgePixels[] = imagecolorsforindex($image, $rgb);
    }
    
    // Calculate average background color
    $avgR = array_sum(array_column($edgePixels, 'red')) / count($edgePixels);
    $avgG = array_sum(array_column($edgePixels, 'green')) / count($edgePixels);
    $avgB = array_sum(array_column($edgePixels, 'blue')) / count($edgePixels);
    
    // Create a new image
    $newImage = imagecreatetruecolor($width, $height);
    imagealphablending($newImage, true);
    
    // Set the new background color
    switch ($newBgType) {
        case 'white':
            $newBgColor = imagecolorallocate($newImage, 255, 255, 255);
            break;
        case 'light':
            $newBgColor = imagecolorallocate($newImage, 245, 245, 245);
            break;
        case 'blue':
            $newBgColor = imagecolorallocate($newImage, 220, 235, 250);
            break;
        default:
            $newBgColor = imagecolorallocate($newImage, 255, 255, 255);
    }
    
    // Process each pixel
    $tolerance = 35; // Adjust for more/less aggressive replacement
    $edgeSoftness = 15; // For smoother edges
    
    for ($y = 0; $y < $height; $y++) {
        for ($x = 0; $x < $width; $x++) {
            $rgb = imagecolorat($image, $x, $y);
            $colors = imagecolorsforindex($image, $rgb);
            
            // Calculate color difference from background
            $diffR = abs($colors['red'] - $avgR);
            $diffG = abs($colors['green'] - $avgG);
            $diffB = abs($colors['blue'] - $avgB);
            $totalDiff = $diffR + $diffG + $diffB;
            
            if ($totalDiff < $tolerance * 3) {
                // This is background, replace it
                imagesetpixel($newImage, $x, $y, $newBgColor);
            } elseif ($totalDiff < ($tolerance + $edgeSoftness) * 3) {
                // This is an edge pixel, blend it
                $alpha = ($totalDiff - $tolerance * 3) / ($edgeSoftness * 3);
                
                // Blend between background and original color
                $blendR = $colors['red'] * $alpha + red($newBgColor) * (1 - $alpha);
                $blendG = $colors['green'] * $alpha + green($newBgColor) * (1 - $alpha);
                $blendB = $colors['blue'] * $alpha + blue($newBgColor) * (1 - $alpha);
                
                $blendedColor = imagecolorallocate($newImage, $blendR, $blendG, $blendB);
                imagesetpixel($newImage, $x, $y, $blendedColor);
            } else {
                // This is foreground, keep it
                imagesetpixel($newImage, $x, $y, $rgb);
            }
        }
    }
    
    // Apply smoothing filter to reduce artifacts
    imagefilter($newImage, IMG_FILTER_SMOOTH, 1);
    
    return $newImage;
}

// Helper functions to extract RGB values
function red($color) {
    return ($color >> 16) & 0xFF;
}

function green($color) {
    return ($color >> 8) & 0xFF;
}

function blue($color) {
    return $color & 0xFF;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Photo Resizer Pro - Edit Your Photo</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1);
            padding: 40px;
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        h1 {
            color: #333;
            text-align: center;
            margin-bottom: 40px;
            font-size: 32px;
            font-weight: 700;
        }
        
        h3 {
            color: #555;
            margin-bottom: 15px;
            font-size: 18px;
            font-weight: 600;
        }
        
        .editor-container {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 40px;
            margin-top: 30px;
        }
        
        .preview-section {
            text-align: center;
        }
        
        .controls-section {
            background: #f8f9fa;
            padding: 30px;
            border-radius: 16px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
        }
        
        .preview-box {
            border: 3px solid #e0e0e0;
            border-radius: 16px;
            padding: 20px;
            background: linear-gradient(45deg, #f5f5f5 25%, transparent 25%, transparent 75%, #f5f5f5 75%, #f5f5f5),
                        linear-gradient(45deg, #f5f5f5 25%, transparent 25%, transparent 75%, #f5f5f5 75%, #f5f5f5);
            background-size: 20px 20px;
            background-position: 0 0, 10px 10px;
            position: relative;
            display: inline-block;
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.1);
        }
        
        #preview {
            max-width: 100%;
            max-height: 400px;
            display: block;
            border-radius: 8px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }
        
        .crop-container {
            position: relative;
            display: inline-block;
            margin-bottom: 30px;
            border-radius: 12px;
            overflow: visible; /* Changed to visible to ensure crop box is not cut off */
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.1);
        }
        
        #sourceImage {
            max-width: 100%;
            max-height: 400px;
            display: block;
            width: auto;
            height: auto;
        }
        
        .crop-box {
            position: absolute;
            border: 3px dashed #667eea;
            background: rgba(102, 126, 234, 0.15);
            cursor: move;
            transition: all 0.3s ease;
        }
        
        .crop-box:hover {
            background: rgba(102, 126, 234, 0.25);
        }
        
        /* Resize handles */
        .resize-handle {
            position: absolute;
            width: 12px;
            height: 12px;
            background: #667eea;
            border: 2px solid white;
            border-radius: 50%;
            cursor: pointer;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
        }
        
        .resize-handle:hover {
            background: #764ba2;
            transform: scale(1.2);
        }
        
        .handle-nw {
            top: -6px;
            left: -6px;
            cursor: nw-resize;
        }
        
        .handle-ne {
            top: -6px;
            right: -6px;
            cursor: ne-resize;
        }
        
        .handle-sw {
            bottom: -6px;
            left: -6px;
            cursor: sw-resize;
        }
        
        .handle-se {
            bottom: -6px;
            right: -6px;
            cursor: se-resize;
        }
        
        /* Edge handles */
        .handle-n {
            top: -6px;
            left: 50%;
            transform: translateX(-50%);
            cursor: n-resize;
        }
        
        .handle-e {
            right: -6px;
            top: 50%;
            transform: translateY(-50%);
            cursor: e-resize;
        }
        
        .handle-s {
            bottom: -6px;
            left: 50%;
            transform: translateX(-50%);
            cursor: s-resize;
        }
        
        .handle-w {
            left: -6px;
            top: 50%;
            transform: translateY(-50%);
            cursor: w-resize;
        }
        
        .control-group {
            margin-bottom: 25px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            color: #444;
            font-weight: 600;
            font-size: 14px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        select, input[type="number"] {
            width: 100%;
            padding: 12px 16px;
            border: 2px solid #e0e0e0;
            border-radius: 10px;
            font-size: 16px;
            transition: all 0.3s ease;
            background: white;
        }
        
        select:focus, input[type="number"]:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        input[type="range"] {
            width: 100%;
            height: 8px;
            background: #e0e0e0;
            outline: none;
            border-radius: 5px;
            -webkit-appearance: none;
            margin: 10px 0;
        }
        
        input[type="range"]::-webkit-slider-thumb {
            -webkit-appearance: none;
            appearance: none;
            width: 20px;
            height: 20px;
            background: #667eea;
            cursor: pointer;
            border-radius: 50%;
            transition: all 0.3s ease;
        }
        
        input[type="range"]::-webkit-slider-thumb:hover {
            background: #764ba2;
            transform: scale(1.2);
        }
        
        .range-value {
            display: inline-block;
            background: #667eea;
            color: white;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            margin-left: 10px;
        }
        
        .button-group {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
            margin-top: 30px;
        }
        
        button {
            padding: 14px 24px;
            border: none;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .primary-btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
        }
        
        .primary-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 20px rgba(102, 126, 234, 0.4);
        }
        
        .secondary-btn {
            background: #f8f9fa;
            color: #666;
            border: 2px solid #e0e0e0;
        }
        
        .secondary-btn:hover {
            background: #e9ecef;
            border-color: #667eea;
            color: #667eea;
        }
        
        .info-box {
            background: linear-gradient(135deg, #e3f2fd 0%, #f3e5f5 100%);
            border: none;
            padding: 20px;
            border-radius: 12px;
            margin-bottom: 30px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
        }
        
        .info-box strong {
            color: #667eea;
            display: block;
            margin-bottom: 10px;
            font-size: 16px;
        }
        
        .warning {
            background: #fff3cd;
            border-left: 4px solid #ffc107;
            color: #856404;
            padding: 15px 20px;
            border-radius: 8px;
            margin-top: 15px;
            display: none;
            animation: slideIn 0.3s ease;
        }
        
        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateX(-20px);
            }
            to {
                opacity: 1;
                transform: translateX(0);
            }
        }
        
        .file-info {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            padding: 20px;
            border-radius: 12px;
            margin-top: 20px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
        }
        
        .file-info strong {
            color: #667eea;
            font-weight: 600;
        }
        
        #loading {
            display: none;
            text-align: center;
            color: #667eea;
            margin: 20px 0;
            font-weight: 600;
        }
        
        #loading::after {
            content: '';
            display: inline-block;
            width: 20px;
            height: 20px;
            margin-left: 10px;
            border: 3px solid #667eea;
            border-radius: 50%;
            border-top-color: transparent;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        /* Progress bar */
        .progress-bar {
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 40px;
            padding: 20px;
        }
        
        .progress-step {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 8px;
        }
        
        .progress-step .step-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #e0e0e0;
            color: #666;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            transition: all 0.3s ease;
        }
        
        .progress-step.completed .step-icon {
            background: #48c774;
            color: white;
        }
        
        .progress-step.active .step-icon {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
        }
        
        .progress-step .step-label {
            font-size: 14px;
            color: #666;
            font-weight: 500;
        }
        
        .progress-line {
            width: 100px;
            height: 2px;
            background: #e0e0e0;
            margin: 0 20px;
        }
        
        /* Section headers */
        .section-header {
            display: flex;
            justify-content: space-between;
            align-items: baseline;
            margin-bottom: 15px;
        }
        
        .help-text {
            font-size: 13px;
            color: #888;
            font-style: italic;
        }
        
        /* Instructions box */
        .instructions-box {
            background: #fafafa;
            border: 1px solid #e0e0e0;
            padding: 20px;
            border-radius: 12px;
            margin-bottom: 30px;
        }
        
        .instructions-box h4 {
            color: #333;
            margin-bottom: 15px;
            font-size: 16px;
        }
        
        .instructions-list {
            margin: 0;
            padding-left: 20px;
            color: #666;
        }
        
        .instructions-list li {
            margin-bottom: 10px;
            line-height: 1.6;
        }
        
        .instructions-list strong {
            color: #444;
        }
        
        /* Info indicators */
        .info-row {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .size-indicator {
            display: inline-block;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            background: #e0e0e0;
        }
        
        .size-indicator.optimal {
            background: #48c774;
            box-shadow: 0 0 0 3px rgba(72, 199, 116, 0.2);
        }
        
        .size-indicator.acceptable {
            background: #ffc107;
            box-shadow: 0 0 0 3px rgba(255, 193, 7, 0.2);
        }
        
        .size-indicator.too-large {
            background: #f44336;
            box-shadow: 0 0 0 3px rgba(244, 67, 54, 0.2);
        }
        
        .success {
            background: #d4edda;
            border-left: 4px solid #28a745;
            color: #155724;
            padding: 15px 20px;
            border-radius: 8px;
            margin-top: 15px;
            animation: slideIn 0.3s ease;
        }
        
        .req-summary {
            line-height: 1.8;
        }
        
        .highlight {
            color: #667eea;
            font-weight: 600;
        }
        
        /* Tooltips */
        .tooltip {
            position: relative;
            display: inline-block;
            margin-left: 5px;
            cursor: help;
        }
        
        .tooltip .tooltip-icon {
            color: #667eea;
            font-size: 14px;
        }
        
        .tooltip .tooltip-text {
            visibility: hidden;
            width: 200px;
            background-color: #333;
            color: #fff;
            text-align: center;
            padding: 10px;
            border-radius: 6px;
            position: absolute;
            z-index: 1;
            bottom: 125%;
            left: 50%;
            margin-left: -100px;
            opacity: 0;
            transition: opacity 0.3s;
            font-size: 13px;
        }
        
        .tooltip:hover .tooltip-text {
            visibility: visible;
            opacity: 1;
        }
        
        @media (max-width: 1024px) {
            .editor-container {
                grid-template-columns: 1fr;
                gap: 30px;
            }
            
            .container {
                padding: 30px 20px;
            }
            
            .progress-line {
                width: 50px;
            }
        }
        
        @media (max-width: 600px) {
            h1 {
                font-size: 24px;
            }
            
            .button-group {
                grid-template-columns: 1fr;
            }
            
            .controls-section {
                padding: 20px;
            }
            
            .info-box {
                font-size: 14px;
            }
            
            .progress-bar {
                padding: 10px;
            }
            
            .progress-step .step-icon {
                width: 35px;
                height: 35px;
                font-size: 14px;
            }
            
            .progress-line {
                width: 30px;
                margin: 0 10px;
            }
            
            .section-header {
                flex-direction: column;
                gap: 5px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>✨ Edit Your Photo</h1>
        
        <div class="progress-bar">
            <div class="progress-step completed">
                <span class="step-icon">✓</span>
                <span class="step-label">Upload</span>
            </div>
            <div class="progress-line"></div>
            <div class="progress-step active">
                <span class="step-icon">2</span>
                <span class="step-label">Edit</span>
            </div>
            <div class="progress-line"></div>
            <div class="progress-step">
                <span class="step-icon">3</span>
                <span class="step-label">Download</span>
            </div>
        </div>
        
        <div class="editor-container">
            <div class="preview-section">
                <div class="section-header">
                    <h3>📸 Original Image</h3>
                    <span class="help-text">Drag the dotted box to crop your photo</span>
                </div>
                <div class="crop-container">
                    <img id="sourceImage" src="data:image/jpeg;base64,<?php echo base64_encode(file_get_contents($imageInfo['path'])); ?>" alt="Original">
                    <div class="crop-box" id="cropBox"></div>
                </div>
                
                <div class="section-header">
                    <h3>👁️ Live Preview</h3>
                    <span class="help-text">Your photo will look like this</span>
                </div>
                <div class="preview-box">
                    <img id="preview" src="data:image/jpeg;base64,<?php echo base64_encode(file_get_contents($imageInfo['path'])); ?>" alt="Preview">
                </div>
                
                <div class="file-info">
                    <div class="info-row">
                        <strong>📊 File Size:</strong> <span id="fileSize">-</span> KB
                        <span class="size-indicator" id="sizeIndicator"></span>
                    </div>
                    <div class="warning" id="sizeWarning"></div>
                    <div class="success" id="sizeSuccess" style="display: none;">
                        ✓ Perfect! File size meets all requirements.
                    </div>
                </div>
            </div>
            
            <div class="controls-section">
                
                <div class="control-group">
                    <label for="presetSize">
                        Size:
                        <span class="tooltip">
                            <span class="tooltip-icon">ⓘ</span>
                            <span class="tooltip-text">Select from common passport/visa sizes or choose custom dimensions</span>
                        </span>
                    </label>
                    <select id="presetSize">
                        <?php foreach ($presetSizes as $label => $size): ?>
                            <option value="<?php echo $size['width'] . 'x' . $size['height']; ?>"><?php echo $label; ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div id="customSize" style="display: none;">
                    <div class="control-group">
                        <label for="customWidth">Width (px):</label>
                        <input type="number" id="customWidth" min="50" max="2000" value="350">
                    </div>
                    <div class="control-group">
                        <label for="customHeight">Height (px):</label>
                        <input type="number" id="customHeight" min="50" max="2000" value="350">
                    </div>
                </div>
                
                <div class="control-group">
                    <label for="brightness">
                        Brightness:
                        <span class="tooltip">
                            <span class="tooltip-icon">ⓘ</span>
                            <span class="tooltip-text">Adjust if photo is too dark or too bright. 0 = no change</span>
                        </span>
                    </label>
                    <input type="range" id="brightness" min="-100" max="100" value="0">
                    <span class="range-value" id="brightnessValue">0</span>
                </div>
                
                <div class="control-group">
                    <label for="contrast">
                        Contrast:
                        <span class="tooltip">
                            <span class="tooltip-icon">ⓘ</span>
                            <span class="tooltip-text">Increase for more definition, decrease for softer look</span>
                        </span>
                    </label>
                    <input type="range" id="contrast" min="-100" max="100" value="0">
                    <span class="range-value" id="contrastValue">0</span>
                </div>
                
                <div class="control-group">
                    <label for="background">
                        Background:
                        <span class="tooltip">
                            <span class="tooltip-icon">ⓘ</span>
                            <span class="tooltip-text">Choose 'Replace' options to change the background color. Works best with solid color backgrounds</span>
                        </span>
                    </label>
                    <select id="background">
                        <option value="keep">Keep Original Background</option>
                        <option value="white">Replace with White</option>
                        <option value="light">Replace with Light Gray</option>
                        <option value="blue">Replace with Light Blue</option>
                    </select>
                </div>
                
                <div class="control-group">
                    <label for="quality">
                        Compression Quality:
                        <span class="tooltip">
                            <span class="tooltip-icon">ⓘ</span>
                            <span class="tooltip-text">Lower values = smaller file size but reduced quality</span>
                        </span>
                    </label>
                    <input type="range" id="quality" min="60" max="100" value="85">
                    <span class="range-value" id="qualityValue">85%</span>
                </div>
                
                <div class="control-group">
                    <label for="dpi">
                        Print Resolution (DPI):
                        <span class="tooltip">
                            <span class="tooltip-icon">ⓘ</span>
                            <span class="tooltip-text">Higher DPI = better print quality. Most documents require 200-300 DPI</span>
                        </span>
                    </label>
                    <select id="dpi">
                        <option value="200">200 DPI (Standard)</option>
                        <option value="250">250 DPI (Better Quality)</option>
                        <option value="300">300 DPI (Best Quality)</option>
                    </select>
                </div>
                
                <div id="loading">Processing...</div>
                
                <div class="button-group">
                    <button class="primary-btn" onclick="updatePreview()">Update Preview</button>
                    <button class="primary-btn" onclick="downloadImage()">Download Photo</button>
                </div>
                
                <div class="button-group">
                    <button class="secondary-btn" onclick="resetSettings()">Reset Settings</button>
                    <button class="secondary-btn" onclick="window.location.href='index.php'">Upload New Image</button>
                </div>
            </div>
        </div>
    </div>

    <script>
        let cropBox = { x: 0, y: 0, width: 200, height: 200 };
        let isDragging = false;
        let isResizing = false;
        let currentHandle = null;
        let dragStart = { x: 0, y: 0 };
        let imageScale = 1;
        let originalWidth = <?php echo $imageInfo['width']; ?>;
        let originalHeight = <?php echo $imageInfo['height']; ?>;
        
        // Initialize
        document.addEventListener('DOMContentLoaded', function() {
            console.log('DOM loaded, initializing...');
            
            // Setup and initialize in sequence
            setTimeout(() => {
                console.log('Setting up crop box...');
                setupCropBox();
                initializeCropBox();
                
                // Add event listeners after initialization
                document.getElementById('presetSize').addEventListener('change', function() {
                    handleSizeChange();
                    updateCropBoxAspectRatio();
                    updatePreview();
                });
                
                document.getElementById('brightness').addEventListener('input', function(e) {
                    updateSliderValue(e);
                    updatePreview();
                });
                
                document.getElementById('contrast').addEventListener('input', function(e) {
                    updateSliderValue(e);
                    updatePreview();
                });
                
                document.getElementById('quality').addEventListener('input', function(e) {
                    updateSliderValue(e);
                    updatePreview();
                });
                
                document.getElementById('background').addEventListener('change', updatePreview);
                
                // Auto-update preview on custom size change
                const customWidth = document.getElementById('customWidth');
                const customHeight = document.getElementById('customHeight');
                
                if (customWidth) {
                    customWidth.addEventListener('change', function() {
                        updateCropBoxAspectRatio();
                        updatePreview();
                    });
                }
                
                if (customHeight) {
                    customHeight.addEventListener('change', function() {
                        updateCropBoxAspectRatio();
                        updatePreview();
                    });
                }
                
                // Initial preview after a short delay
                setTimeout(updatePreview, 500);
            }, 100);
        });
        
        function setupCropBox() {
            const cropElement = document.getElementById('cropBox');
            
            // Add corner and edge resize handles
            const handles = ['nw', 'n', 'ne', 'e', 'se', 's', 'sw', 'w'];
            handles.forEach(handle => {
                const handleElement = document.createElement('div');
                handleElement.className = 'resize-handle handle-' + handle;
                handleElement.dataset.handle = handle;
                cropElement.appendChild(handleElement);
            });
            
            // Event listeners
            cropElement.addEventListener('mousedown', startDrag);
            document.querySelectorAll('.resize-handle').forEach(handle => {
                handle.addEventListener('mousedown', startResize);
            });
            document.addEventListener('mousemove', handleMouseMove);
            document.addEventListener('mouseup', handleMouseUp);
        }
        
        function initializeCropBox() {
            const img = document.getElementById('sourceImage');
            // Wait for image to load completely
            if (img.complete && img.naturalHeight !== 0) {
                setCropBoxInitialSize();
            } else {
                img.onload = function() {
                    setCropBoxInitialSize();
                };
            }
        }
        
        function setCropBoxInitialSize() {
            const img = document.getElementById('sourceImage');
            console.log('Setting initial crop box size. Image dimensions:', img.offsetWidth, 'x', img.offsetHeight);
            
            // Calculate the scale between displayed size and actual size
            imageScale = img.naturalWidth / img.offsetWidth;
            
            // Get the target aspect ratio
            const preset = document.getElementById('presetSize').value;
            let targetWidth, targetHeight;
            
            if (preset === '0x0') {
                targetWidth = parseInt(document.getElementById('customWidth')?.value) || 350;
                targetHeight = parseInt(document.getElementById('customHeight')?.value) || 350;
            } else {
                [targetWidth, targetHeight] = preset.split('x').map(v => parseInt(v));
            }
            
            const targetAspectRatio = targetWidth / targetHeight;
            
            // Calculate the largest possible crop box with the target aspect ratio
            const imgWidth = img.offsetWidth;
            const imgHeight = img.offsetHeight;
            const imgAspectRatio = imgWidth / imgHeight;
            
            if (imgAspectRatio > targetAspectRatio) {
                // Image is wider than target - fit by height
                cropBox.height = imgHeight;
                cropBox.width = imgHeight * targetAspectRatio;
                cropBox.x = (imgWidth - cropBox.width) / 2;
                cropBox.y = 0;
            } else {
                // Image is taller than target - fit by width
                cropBox.width = imgWidth;
                cropBox.height = imgWidth / targetAspectRatio;
                cropBox.x = 0;
                cropBox.y = (imgHeight - cropBox.height) / 2;
            }
            
            console.log('Crop box dimensions:', cropBox);
            updateCropBox();
        }
        
        function updateCropBoxAspectRatio() {
            const preset = document.getElementById('presetSize').value;
            let targetWidth, targetHeight;
            
            if (preset === '0x0') {
                targetWidth = parseInt(document.getElementById('customWidth').value) || 350;
                targetHeight = parseInt(document.getElementById('customHeight').value) || 350;
            } else {
                [targetWidth, targetHeight] = preset.split('x').map(v => parseInt(v));
            }
            
            const aspectRatio = targetWidth / targetHeight;
            const img = document.getElementById('sourceImage');
            const imgWidth = img.offsetWidth;
            const imgHeight = img.offsetHeight;
            
            // For initial setup, try to maintain current crop area if possible
            const currentAspectRatio = cropBox.width / cropBox.height;
            
            if (Math.abs(currentAspectRatio - aspectRatio) > 0.01) {
                // Aspect ratio changed, adjust the crop box
                // Try to maintain the current center point
                const centerX = cropBox.x + cropBox.width / 2;
                const centerY = cropBox.y + cropBox.height / 2;
                
                // Calculate new dimensions maintaining aspect ratio
                // Keep the smaller dimension to ensure we don't exceed image bounds
                const maxWidth = Math.min(imgWidth, cropBox.width);
                const maxHeight = Math.min(imgHeight, cropBox.height);
                
                if (maxWidth / aspectRatio <= imgHeight && maxWidth / aspectRatio <= maxHeight) {
                    cropBox.width = maxWidth;
                    cropBox.height = maxWidth / aspectRatio;
                } else {
                    cropBox.height = maxHeight;
                    cropBox.width = maxHeight * aspectRatio;
                }
                
                // Recenter the crop box
                cropBox.x = centerX - cropBox.width / 2;
                cropBox.y = centerY - cropBox.height / 2;
                
                // Ensure crop box stays within bounds
                cropBox.x = Math.max(0, Math.min(imgWidth - cropBox.width, cropBox.x));
                cropBox.y = Math.max(0, Math.min(imgHeight - cropBox.height, cropBox.y));
            }
            
            updateCropBox();
        }
        
        function updateCropBox() {
            const cropElement = document.getElementById('cropBox');
            cropElement.style.left = cropBox.x + 'px';
            cropElement.style.top = cropBox.y + 'px';
            cropElement.style.width = cropBox.width + 'px';
            cropElement.style.height = cropBox.height + 'px';
        }
        
        function startDrag(e) {
            if (e.target.classList.contains('resize-handle')) return;
            
            const img = document.getElementById('sourceImage');
            const imgRect = img.getBoundingClientRect();
            
            isDragging = true;
            // Store the offset from the mouse to the crop box corner
            dragStart.x = e.clientX - imgRect.left - cropBox.x;
            dragStart.y = e.clientY - imgRect.top - cropBox.y;
            e.preventDefault();
        }
        
        function startResize(e) {
            isResizing = true;
            currentHandle = e.target.dataset.handle;
            dragStart.x = e.clientX;
            dragStart.y = e.clientY;
            dragStart.width = cropBox.width;
            dragStart.height = cropBox.height;
            dragStart.cropX = cropBox.x;
            dragStart.cropY = cropBox.y;
            e.stopPropagation();
            e.preventDefault();
        }
        
        function handleMouseMove(e) {
            if (isDragging) {
                drag(e);
            } else if (isResizing) {
                resize(e);
            }
        }
        
        function handleMouseUp() {
            if (isDragging || isResizing) {
                isDragging = false;
                isResizing = false;
                currentHandle = null;
                updatePreview();
            }
        }
        
        function drag(e) {
            const img = document.getElementById('sourceImage');
            const imgRect = img.getBoundingClientRect();
            
            // Calculate new position relative to the image
            const newX = e.clientX - imgRect.left - dragStart.x;
            const newY = e.clientY - imgRect.top - dragStart.y;
            
            // Constrain to image bounds
            cropBox.x = Math.max(0, Math.min(img.offsetWidth - cropBox.width, newX));
            cropBox.y = Math.max(0, Math.min(img.offsetHeight - cropBox.height, newY));
            
            updateCropBox();
        }
        
        function resize(e) {
            const img = document.getElementById('sourceImage');
            const imgRect = img.getBoundingClientRect();
            const dx = e.clientX - dragStart.x;
            const dy = e.clientY - dragStart.y;
            
            const preset = document.getElementById('presetSize').value;
            let targetWidth, targetHeight;
            
            if (preset === '0x0') {
                targetWidth = parseInt(document.getElementById('customWidth').value) || 350;
                targetHeight = parseInt(document.getElementById('customHeight').value) || 350;
            } else {
                [targetWidth, targetHeight] = preset.split('x').map(v => parseInt(v));
            }
            
            const aspectRatio = targetWidth / targetHeight;
            
            // Calculate new dimensions based on which handle is being dragged
            let newX = cropBox.x;
            let newY = cropBox.y;
            let newWidth = cropBox.width;
            let newHeight = cropBox.height;
            
            switch (currentHandle) {
                // Corner handles - maintain aspect ratio
                case 'se': // Bottom-right
                    const seDelta = Math.max(Math.abs(dx), Math.abs(dy) * aspectRatio);
                    newWidth = Math.max(50, dragStart.width + seDelta * Math.sign(dx));
                    newHeight = newWidth / aspectRatio;
                    break;
                    
                case 'sw': // Bottom-left
                    const swDelta = Math.max(Math.abs(dx), Math.abs(dy) * aspectRatio);
                    newWidth = Math.max(50, dragStart.width + swDelta * Math.sign(-dx));
                    newHeight = newWidth / aspectRatio;
                    newX = dragStart.cropX + dragStart.width - newWidth;
                    break;
                    
                case 'ne': // Top-right
                    const neDelta = Math.max(Math.abs(dx), Math.abs(dy) * aspectRatio);
                    newWidth = Math.max(50, dragStart.width + neDelta * Math.sign(dx));
                    newHeight = newWidth / aspectRatio;
                    newY = dragStart.cropY + dragStart.height - newHeight;
                    break;
                    
                case 'nw': // Top-left
                    const nwDelta = Math.max(Math.abs(dx), Math.abs(dy) * aspectRatio);
                    newWidth = Math.max(50, dragStart.width + nwDelta * Math.sign(-dx));
                    newHeight = newWidth / aspectRatio;
                    newX = dragStart.cropX + dragStart.width - newWidth;
                    newY = dragStart.cropY + dragStart.height - newHeight;
                    break;
                    
                // Edge handles - adjust single dimension and maintain aspect ratio
                case 'n': // Top
                    newHeight = Math.max(50, dragStart.height - dy);
                    newWidth = newHeight * aspectRatio;
                    newY = dragStart.cropY + dragStart.height - newHeight;
                    newX = dragStart.cropX + (dragStart.width - newWidth) / 2;
                    break;
                    
                case 'e': // Right
                    newWidth = Math.max(50, dragStart.width + dx);
                    newHeight = newWidth / aspectRatio;
                    newY = dragStart.cropY + (dragStart.height - newHeight) / 2;
                    break;
                    
                case 's': // Bottom
                    newHeight = Math.max(50, dragStart.height + dy);
                    newWidth = newHeight * aspectRatio;
                    newX = dragStart.cropX + (dragStart.width - newWidth) / 2;
                    break;
                    
                case 'w': // Left
                    newWidth = Math.max(50, dragStart.width - dx);
                    newHeight = newWidth / aspectRatio;
                    newX = dragStart.cropX + dragStart.width - newWidth;
                    newY = dragStart.cropY + (dragStart.height - newHeight) / 2;
                    break;
            }
            
            // Ensure crop box stays within image bounds
            if (newX < 0) {
                newWidth += newX;
                newHeight = newWidth / aspectRatio;
                newX = 0;
            }
            if (newY < 0) {
                newHeight += newY;
                newWidth = newHeight * aspectRatio;
                newY = 0;
            }
            if (newX + newWidth > img.offsetWidth) {
                newWidth = img.offsetWidth - newX;
                newHeight = newWidth / aspectRatio;
            }
            if (newY + newHeight > img.offsetHeight) {
                newHeight = img.offsetHeight - newY;
                newWidth = newHeight * aspectRatio;
            }
            
            // Apply the new dimensions
            cropBox.x = newX;
            cropBox.y = newY;
            cropBox.width = Math.max(50, newWidth);
            cropBox.height = Math.max(50 / aspectRatio, newHeight);
            
            updateCropBox();
        }
        
        function stopDrag() {
            isDragging = false;
            isResizing = false;
            currentHandle = null;
        }
        
        function handleSizeChange() {
            const preset = document.getElementById('presetSize').value;
            const customDiv = document.getElementById('customSize');
            
            if (preset === '0x0') {
                customDiv.style.display = 'block';
            } else {
                customDiv.style.display = 'none';
            }
        }
        
        function updateSliderValue(e) {
            const valueSpan = document.getElementById(e.target.id + 'Value');
            if (e.target.id === 'quality') {
                valueSpan.textContent = e.target.value + '%';
            } else {
                valueSpan.textContent = e.target.value;
            }
        }
        
        function getSettings() {
            const preset = document.getElementById('presetSize').value;
            let width, height;
            
            if (preset === '0x0') {
                width = document.getElementById('customWidth').value;
                height = document.getElementById('customHeight').value;
            } else {
                [width, height] = preset.split('x');
            }
            
            const img = document.getElementById('sourceImage');
            // Recalculate scale in case image size changed
            const scale = img.naturalWidth / img.offsetWidth;
            
            return {
                width: parseInt(width),
                height: parseInt(height),
                cropX: Math.round(cropBox.x * scale),
                cropY: Math.round(cropBox.y * scale),
                cropWidth: Math.round(cropBox.width * scale),
                cropHeight: Math.round(cropBox.height * scale),
                brightness: document.getElementById('brightness').value,
                contrast: document.getElementById('contrast').value,
                background: document.getElementById('background').value,
                quality: document.getElementById('quality').value,
                dpi: document.getElementById('dpi').value
            };
        }
        
        function updatePreview() {
            document.getElementById('loading').style.display = 'block';
            
            const settings = getSettings();
            settings.action = 'preview';
            
            fetch('process.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                    'X-Requested-With': 'XMLHttpRequest'
                },
                body: new URLSearchParams(settings)
            })
            .then(response => response.json())
            .then(data => {
                document.getElementById('loading').style.display = 'none';
                
                if (data.success) {
                    document.getElementById('preview').src = data.preview;
                    document.getElementById('fileSize').textContent = data.fileSizeKB;
                    
                    // Update size indicator
                    const sizeIndicator = document.getElementById('sizeIndicator');
                    const successMsg = document.getElementById('sizeSuccess');
                    const warning = document.getElementById('sizeWarning');
                    
                    if (data.fileSize >= 20480 && data.fileSize <= 102400) {
                        // Optimal size (20-100KB)
                        sizeIndicator.className = 'size-indicator optimal';
                        successMsg.style.display = 'block';
                        warning.style.display = 'none';
                    } else if (data.fileSize > 102400 && data.fileSize <= 307200) {
                        // Acceptable size (100-300KB)
                        sizeIndicator.className = 'size-indicator acceptable';
                        warning.textContent = 'File size is acceptable but could be optimized. Try increasing compression.';
                        warning.style.display = 'block';
                        successMsg.style.display = 'none';
                    } else if (data.fileSize < 20480) {
                        // Too small
                        sizeIndicator.className = 'size-indicator too-large';
                        warning.textContent = 'File size is too small. Try reducing compression quality.';
                        warning.style.display = 'block';
                        successMsg.style.display = 'none';
                    } else {
                        // Too large
                        sizeIndicator.className = 'size-indicator too-large';
                        warning.textContent = 'File size exceeds 300KB limit. Increase compression or reduce dimensions.';
                        warning.style.display = 'block';
                        successMsg.style.display = 'none';
                    }
                } else {
                    alert('Error: ' + data.error);
                }
            })
            .catch(error => {
                document.getElementById('loading').style.display = 'none';
                alert('Error updating preview');
            });
        }
        
        function downloadImage() {
            const settings = getSettings();
            settings.action = 'download';
            
            fetch('process.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                    'X-Requested-With': 'XMLHttpRequest'
                },
                body: new URLSearchParams(settings)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    window.location.href = data.downloadUrl;
                } else {
                    alert('Error: ' + data.error);
                }
            })
            .catch(error => {
                alert('Error downloading image');
            });
        }
        
        function resetSettings() {
            document.getElementById('brightness').value = 0;
            document.getElementById('contrast').value = 0;
            document.getElementById('quality').value = 85;
            document.getElementById('background').value = 'white';
            updateSliderValue({target: document.getElementById('brightness')});
            updateSliderValue({target: document.getElementById('contrast')});
            updateSliderValue({target: document.getElementById('quality')});
            initializeCropBox();
            updatePreview();
        }
    </script>
</body>
</html>