<?php
session_start();

// Check if image was uploaded
if (!isset($_SESSION['uploaded_image']) || !file_exists($_SESSION['uploaded_image']['path'])) {
    header('Location: index.php');
    exit;
}

$imageInfo = $_SESSION['uploaded_image'];

// Preset sizes with labels
$presetSizes = [
    '3.5x4.5cm (413x531px)' => ['width' => 413, 'height' => 531],
    '240x320px' => ['width' => 240, 'height' => 320],
    '200x230px' => ['width' => 200, 'height' => 230],
    '350x350px' => ['width' => 350, 'height' => 350],
    '600x600px' => ['width' => 600, 'height' => 600],
    '800x800px' => ['width' => 800, 'height' => 800],
    'Custom' => ['width' => 0, 'height' => 0]
];

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Photo Resizer Pro - Edit Your Photo</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
            background: white;
            border-radius: 20px;
            padding: 40px;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1);
        }
        
        h1 {
            text-align: center;
            color: #333;
            margin-bottom: 10px;
            font-size: 2.5em;
        }
        
        .subtitle {
            text-align: center;
            color: #666;
            margin-bottom: 30px;
        }
        
        .main-content {
            display: grid;
            grid-template-columns: 1fr 350px;
            gap: 40px;
        }
        
        .image-section {
            background: #f8f9fa;
            padding: 30px;
            border-radius: 12px;
        }
        
        .image-container {
            position: relative;
            display: inline-block;
            margin: 0 auto;
            background: #ddd;
            user-select: none;
        }
        
        #sourceImage {
            display: block;
            max-width: 100%;
            height: auto;
            max-height: 500px;
        }
        
        .crop-box {
            position: absolute;
            border: 2px solid #007bff;
            background: rgba(0, 123, 255, 0.2);
            cursor: move;
            display: none;
        }
        
        .crop-handle {
            position: absolute;
            width: 12px;
            height: 12px;
            background: #007bff;
            border: 2px solid white;
            box-shadow: 0 0 3px rgba(0,0,0,0.5);
            border-radius: 50%;
        }
        
        .crop-handle.tl { top: -6px; left: -6px; cursor: nw-resize; }
        .crop-handle.tr { top: -6px; right: -6px; cursor: ne-resize; }
        .crop-handle.bl { bottom: -6px; left: -6px; cursor: sw-resize; }
        .crop-handle.br { bottom: -6px; right: -6px; cursor: se-resize; }
        .crop-handle.tm { top: -6px; left: 50%; transform: translateX(-50%); cursor: n-resize; }
        .crop-handle.bm { bottom: -6px; left: 50%; transform: translateX(-50%); cursor: s-resize; }
        .crop-handle.ml { left: -6px; top: 50%; transform: translateY(-50%); cursor: w-resize; }
        .crop-handle.mr { right: -6px; top: 50%; transform: translateY(-50%); cursor: e-resize; }
        
        .controls-section {
            background: #f8f9fa;
            padding: 30px;
            border-radius: 12px;
        }
        
        .control-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #444;
            font-size: 14px;
        }
        
        select, input[type="number"], input[type="range"] {
            width: 100%;
            padding: 10px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        
        select:focus, input[type="number"]:focus {
            outline: none;
            border-color: #667eea;
        }
        
        input[type="range"] {
            padding: 5px 0;
        }
        
        .range-value {
            display: inline-block;
            margin-left: 10px;
            font-weight: 600;
            color: #667eea;
        }
        
        .background-options {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 10px;
            margin-top: 10px;
        }
        
        .bg-option {
            padding: 10px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            cursor: pointer;
            text-align: center;
            transition: all 0.3s;
        }
        
        .bg-option:hover {
            border-color: #667eea;
        }
        
        .bg-option.active {
            border-color: #667eea;
            background: #f0f4ff;
        }
        
        .preview-section {
            margin-top: 30px;
            padding: 20px;
            background: white;
            border-radius: 12px;
            text-align: center;
        }
        
        #preview {
            max-width: 100%;
            max-height: 300px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            margin: 10px 0;
        }
        
        .file-info {
            display: flex;
            justify-content: center;
            gap: 30px;
            margin-top: 15px;
        }
        
        .info-item {
            text-align: center;
        }
        
        .info-label {
            font-size: 12px;
            color: #666;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .info-value {
            font-size: 18px;
            font-weight: 600;
            color: #333;
        }
        
        .size-indicator {
            display: inline-block;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            margin-left: 5px;
        }
        
        .size-optimal { background: #28a745; }
        .size-acceptable { background: #ffc107; }
        .size-large { background: #dc3545; }
        
        .button-group {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 10px;
            margin-top: 20px;
        }
        
        button {
            padding: 12px 20px;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .primary-btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .primary-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
        }
        
        .secondary-btn {
            background: #e0e0e0;
            color: #333;
        }
        
        .secondary-btn:hover {
            background: #d0d0d0;
        }
        
        @media (max-width: 1024px) {
            .main-content {
                grid-template-columns: 1fr;
            }
            
            .controls-section {
                max-width: 600px;
                margin: 0 auto;
            }
        }
        
        @media (max-width: 768px) {
            .container {
                padding: 20px;
            }
            
            h1 {
                font-size: 2em;
            }
            
            .file-info {
                flex-direction: column;
                gap: 15px;
            }
        }
        
        .loading {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            justify-content: center;
            align-items: center;
            z-index: 1000;
        }
        
        .loading.active {
            display: flex;
        }
        
        .spinner {
            width: 50px;
            height: 50px;
            border: 5px solid #f3f3f3;
            border-top: 5px solid #667eea;
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        /* Rate limit error styles - Trendy Modal */
        .rate-limit-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.8);
            backdrop-filter: blur(10px);
            z-index: 1000;
            display: none;
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        .rate-limit-overlay.active {
            display: block;
            opacity: 1;
        }
        
        .rate-limit-error {
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%) scale(0.9);
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            padding: 0;
            border-radius: 24px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.5), 0 0 100px rgba(102, 126, 234, 0.3);
            max-width: 420px;
            width: 90%;
            z-index: 1001;
            display: none;
            opacity: 0;
            transition: all 0.3s cubic-bezier(0.68, -0.55, 0.265, 1.55);
            overflow: hidden;
        }
        
        .rate-limit-error.active {
            display: block;
            opacity: 1;
            transform: translate(-50%, -50%) scale(1);
        }
        
        .rate-limit-header {
            background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            padding: 30px;
            text-align: center;
            position: relative;
            overflow: hidden;
        }
        
        .rate-limit-header::before {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
            animation: shimmer 3s linear infinite;
        }
        
        @keyframes shimmer {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .rate-limit-icon {
            font-size: 80px;
            margin-bottom: 15px;
            display: inline-block;
            animation: bounce 2s ease-in-out infinite;
            filter: drop-shadow(0 5px 15px rgba(0,0,0,0.3));
        }
        
        @keyframes bounce {
            0%, 100% { transform: translateY(0); }
            50% { transform: translateY(-10px); }
        }
        
        .rate-limit-title {
            font-size: 28px;
            color: white;
            margin: 0;
            font-weight: 700;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
            position: relative;
        }
        
        .rate-limit-body {
            padding: 40px;
            color: white;
            text-align: center;
        }
        
        .rate-limit-message {
            color: #b8bec8;
            margin-bottom: 30px;
            line-height: 1.6;
            font-size: 16px;
        }
        
        .countdown-container {
            background: rgba(255, 255, 255, 0.1);
            border: 2px solid rgba(255, 255, 255, 0.2);
            border-radius: 16px;
            padding: 25px;
            margin: 20px 0;
            backdrop-filter: blur(10px);
            position: relative;
            overflow: hidden;
        }
        
        .countdown-container::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(45deg, transparent 30%, rgba(255,255,255,0.1) 50%, transparent 70%);
            animation: shine 3s linear infinite;
        }
        
        @keyframes shine {
            0% { transform: translateX(-100%); }
            100% { transform: translateX(100%); }
        }
        
        .countdown-label {
            font-size: 14px;
            color: #b8bec8;
            text-transform: uppercase;
            letter-spacing: 2px;
            margin-bottom: 10px;
        }
        
        .countdown-timer {
            font-size: 56px;
            font-weight: 700;
            color: #f5576c;
            margin: 10px 0;
            font-family: 'Monaco', 'Courier New', monospace;
            text-shadow: 0 0 30px rgba(245, 87, 108, 0.5);
            position: relative;
            z-index: 1;
        }
        
        .countdown-timer span {
            display: inline-block;
            animation: pulse-number 1s ease-in-out infinite;
        }
        
        @keyframes pulse-number {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.05); }
        }
        
        .rate-limit-info {
            font-size: 13px;
            color: #7f8690;
            margin-top: 20px;
            padding: 15px;
            background: rgba(255, 255, 255, 0.05);
            border-radius: 12px;
            border: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .rate-limit-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
            margin-top: 30px;
        }
        
        .action-button {
            padding: 14px 28px;
            border: none;
            border-radius: 12px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            position: relative;
            overflow: hidden;
        }
        
        .action-button::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 0;
            height: 0;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.2);
            transform: translate(-50%, -50%);
            transition: width 0.6s, height 0.6s;
        }
        
        .action-button:hover::before {
            width: 300px;
            height: 300px;
        }
        
        .action-button.primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            box-shadow: 0 5px 20px rgba(102, 126, 234, 0.4);
        }
        
        .action-button.primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.5);
        }
        
        .action-button.secondary {
            background: rgba(255, 255, 255, 0.1);
            color: white;
            border: 2px solid rgba(255, 255, 255, 0.2);
        }
        
        .action-button.secondary:hover {
            background: rgba(255, 255, 255, 0.15);
            border-color: rgba(255, 255, 255, 0.3);
        }
        
        .progress-ring {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            width: 120px;
            height: 120px;
            z-index: 0;
        }
        
        .progress-ring svg {
            transform: rotate(-90deg);
        }
        
        .progress-ring circle {
            fill: none;
            stroke-width: 4;
        }
        
        .progress-ring .bg {
            stroke: rgba(255, 255, 255, 0.1);
        }
        
        .progress-ring .progress {
            stroke: url(#gradient);
            stroke-linecap: round;
            transition: stroke-dashoffset 1s linear;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>✨ Edit Your Photo</h1>
        <p class="subtitle">Crop, resize, and enhance your image</p>
        
        <div class="main-content">
            <div class="image-section">
                <div class="image-container" id="imageContainer">
                    <img id="sourceImage" src="data:image/jpeg;base64,<?php echo base64_encode(file_get_contents($imageInfo['path'])); ?>" alt="Source Image">
                    <div class="crop-box" id="cropBox">
                        <div class="crop-handle tl"></div>
                        <div class="crop-handle tr"></div>
                        <div class="crop-handle bl"></div>
                        <div class="crop-handle br"></div>
                        <div class="crop-handle tm"></div>
                        <div class="crop-handle bm"></div>
                        <div class="crop-handle ml"></div>
                        <div class="crop-handle mr"></div>
                    </div>
                </div>
            </div>
            
            <div class="controls-section">
                <div class="control-group">
                    <label for="presetSize">📐 Size Preset</label>
                    <select id="presetSize">
                        <?php foreach ($presetSizes as $label => $size): ?>
                            <option value="<?php echo $size['width'] . 'x' . $size['height']; ?>"><?php echo $label; ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div id="customSize" style="display: none;">
                    <div class="control-group">
                        <label for="customWidth">Width (px)</label>
                        <input type="number" id="customWidth" min="50" max="2000" value="300">
                    </div>
                    <div class="control-group">
                        <label for="customHeight">Height (px)</label>
                        <input type="number" id="customHeight" min="50" max="2000" value="300">
                    </div>
                </div>
                
                <div class="control-group">
                    <label>🎨 Background</label>
                    <div class="background-options">
                        <div class="bg-option active" data-bg="none">None</div>
                        <div class="bg-option" data-bg="white">White</div>
                        <div class="bg-option" data-bg="gray">Light Gray</div>
                    </div>
                </div>
                
                <div class="control-group">
                    <label for="brightness">☀️ Brightness <span class="range-value" id="brightnessValue">0</span></label>
                    <input type="range" id="brightness" min="-100" max="100" value="0">
                </div>
                
                <div class="control-group">
                    <label for="contrast">🎛️ Contrast <span class="range-value" id="contrastValue">0</span></label>
                    <input type="range" id="contrast" min="-100" max="100" value="0">
                </div>
                
                <div class="control-group">
                    <label for="quality">📊 Quality <span class="range-value" id="qualityValue">85</span>%</label>
                    <input type="range" id="quality" min="60" max="100" value="85" step="5">
                </div>
                
                <div class="preview-section">
                    <h3>Preview</h3>
                    <img id="preview" src="data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='300' height='300'%3E%3Crect width='300' height='300' fill='%23f0f0f0'/%3E%3Ctext x='50%25' y='50%25' text-anchor='middle' dy='.3em' fill='%23999' font-family='Arial' font-size='14'%3EPreview will appear here%3C/text%3E%3C/svg%3E" alt="Preview">
                    <div class="file-info">
                        <div class="info-item">
                            <div class="info-label">Size</div>
                            <div class="info-value"><span id="previewSize">-</span></div>
                        </div>
                        <div class="info-item">
                            <div class="info-label">File Size</div>
                            <div class="info-value">
                                <span id="fileSize">-</span> KB
                                <span class="size-indicator" id="sizeIndicator"></span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="button-group">
                    <button class="secondary-btn" onclick="resetSettings()">🔄 Reset</button>
                    <button class="primary-btn" onclick="downloadImage()">💾 Download</button>
                </div>
                
                <div style="margin-top: 20px; text-align: center;">
                    <a href="index.php" style="color: #667eea; text-decoration: none;">← Upload Another Image</a>
                </div>
            </div>
        </div>
    </div>
    
    <div class="loading" id="loading">
        <div class="spinner"></div>
    </div>
    
    <div class="rate-limit-overlay" id="rateLimitOverlay"></div>
    
    <div class="rate-limit-error" id="rateLimitError">
        <div class="rate-limit-header">
            <div class="rate-limit-icon">⚡</div>
            <h2 class="rate-limit-title">Whoa, Slow Down!</h2>
        </div>
        
        <div class="rate-limit-body">
            <p class="rate-limit-message" id="rateLimitMessage">
                You've hit our rate limit. Don't worry, you can try again soon!
            </p>
            
            <div class="countdown-container">
                <div class="progress-ring">
                    <svg width="120" height="120">
                        <defs>
                            <linearGradient id="gradient" x1="0%" y1="0%" x2="100%" y2="100%">
                                <stop offset="0%" style="stop-color:#f093fb;stop-opacity:1" />
                                <stop offset="100%" style="stop-color:#f5576c;stop-opacity:1" />
                            </linearGradient>
                        </defs>
                        <circle class="bg" cx="60" cy="60" r="54" />
                        <circle class="progress" id="progressCircle" cx="60" cy="60" r="54" />
                    </svg>
                </div>
                <div class="countdown-label">Time Remaining</div>
                <div class="countdown-timer" id="countdownTimer">
                    <span>00</span>:<span>00</span>
                </div>
            </div>
            
            <div class="rate-limit-info">
                💡 <strong>Pro tip:</strong> Our limits reset automatically. Current limits: 2 requests per minute, 5 per hour.
            </div>
            
            <div class="rate-limit-actions">
                <button class="action-button secondary" onclick="closeRateLimitModal()">
                    <span>✕</span> Close
                </button>
                <a href="index.php" class="action-button primary">
                    <span>📸</span> New Upload
                </a>
            </div>
        </div>
    </div>

    <script>
        // Global variables
        let cropData = {
            x: 0,
            y: 0,
            width: 200,
            height: 200
        };
        
        let imageData = {
            width: 0,
            height: 0,
            displayWidth: 0,
            displayHeight: 0,
            scale: 1
        };
        
        let dragState = {
            active: false,
            element: null,
            startX: 0,
            startY: 0,
            startCropX: 0,
            startCropY: 0,
            startCropWidth: 0,
            startCropHeight: 0
        };
        
        let currentBackground = 'none';
        let previewTimer = null;
        let countdownInterval = null;
        
        // Initialize
        window.addEventListener('load', function() {
            const img = document.getElementById('sourceImage');
            img.onload = function() {
                initializeImage();
                initializeCrop();
                setupEventHandlers();
                updateCropBox();
                updatePreview();
            };
            
            if (img.complete) {
                img.onload();
            }
        });
        
        function initializeImage() {
            const img = document.getElementById('sourceImage');
            const container = document.getElementById('imageContainer');
            
            imageData.width = img.naturalWidth;
            imageData.height = img.naturalHeight;
            imageData.displayWidth = img.offsetWidth;
            imageData.displayHeight = img.offsetHeight;
            imageData.scale = img.naturalWidth / img.offsetWidth;
            
            container.style.width = img.offsetWidth + 'px';
            container.style.height = img.offsetHeight + 'px';
        }
        
        function initializeCrop() {
            const img = document.getElementById('sourceImage');
            
            // Initialize with full image selected
            cropData.width = img.offsetWidth;
            cropData.height = img.offsetHeight;
            cropData.x = 0;
            cropData.y = 0;
            
            document.getElementById('cropBox').style.display = 'block';
        }
        
        function setupEventHandlers() {
            const cropBox = document.getElementById('cropBox');
            const handles = cropBox.querySelectorAll('.crop-handle');
            
            // Crop box drag
            cropBox.addEventListener('mousedown', function(e) {
                if (e.target === cropBox) {
                    startDrag(e, 'move');
                }
            });
            
            // Handle drag
            handles.forEach(handle => {
                handle.addEventListener('mousedown', function(e) {
                    const classes = this.className.split(' ');
                    startDrag(e, classes[1]);
                    e.stopPropagation();
                });
            });
            
            // Global mouse events
            document.addEventListener('mousemove', handleDrag);
            document.addEventListener('mouseup', stopDrag);
            
            // Control events
            document.getElementById('presetSize').addEventListener('change', handlePresetChange);
            
            ['customWidth', 'customHeight'].forEach(id => {
                document.getElementById(id).addEventListener('input', function() {
                    clearTimeout(previewTimer);
                    previewTimer = setTimeout(updatePreview, 500);
                });
            });
            
            // Background options
            document.querySelectorAll('.bg-option').forEach(option => {
                option.addEventListener('click', function() {
                    document.querySelectorAll('.bg-option').forEach(o => o.classList.remove('active'));
                    this.classList.add('active');
                    currentBackground = this.dataset.bg;
                    updatePreview();
                });
            });
            
            // Sliders
            ['brightness', 'contrast', 'quality'].forEach(id => {
                document.getElementById(id).addEventListener('input', function() {
                    document.getElementById(id + 'Value').textContent = this.value;
                    if (id !== 'quality') {
                        clearTimeout(previewTimer);
                        previewTimer = setTimeout(updatePreview, 100);
                    }
                });
            });
        }
        
        function startDrag(e, type) {
            dragState.active = true;
            dragState.element = type;
            dragState.startX = e.clientX;
            dragState.startY = e.clientY;
            dragState.startCropX = cropData.x;
            dragState.startCropY = cropData.y;
            dragState.startCropWidth = cropData.width;
            dragState.startCropHeight = cropData.height;
            
            e.preventDefault();
            document.body.style.cursor = e.target.style.cursor || 'move';
        }
        
        function handleDrag(e) {
            if (!dragState.active) return;
            
            const dx = e.clientX - dragState.startX;
            const dy = e.clientY - dragState.startY;
            const img = document.getElementById('sourceImage');
            
            switch(dragState.element) {
                case 'move':
                    cropData.x = Math.max(0, Math.min(img.offsetWidth - cropData.width, dragState.startCropX + dx));
                    cropData.y = Math.max(0, Math.min(img.offsetHeight - cropData.height, dragState.startCropY + dy));
                    break;
                
                case 'tl':
                    cropData.x = Math.max(0, Math.min(dragState.startCropX + dragState.startCropWidth - 50, dragState.startCropX + dx));
                    cropData.y = Math.max(0, Math.min(dragState.startCropY + dragState.startCropHeight - 50, dragState.startCropY + dy));
                    cropData.width = dragState.startCropWidth - (cropData.x - dragState.startCropX);
                    cropData.height = dragState.startCropHeight - (cropData.y - dragState.startCropY);
                    break;
                
                case 'tr':
                    cropData.y = Math.max(0, Math.min(dragState.startCropY + dragState.startCropHeight - 50, dragState.startCropY + dy));
                    cropData.width = Math.max(50, Math.min(img.offsetWidth - cropData.x, dragState.startCropWidth + dx));
                    cropData.height = dragState.startCropHeight - (cropData.y - dragState.startCropY);
                    break;
                
                case 'bl':
                    cropData.x = Math.max(0, Math.min(dragState.startCropX + dragState.startCropWidth - 50, dragState.startCropX + dx));
                    cropData.width = dragState.startCropWidth - (cropData.x - dragState.startCropX);
                    cropData.height = Math.max(50, Math.min(img.offsetHeight - cropData.y, dragState.startCropHeight + dy));
                    break;
                
                case 'br':
                    cropData.width = Math.max(50, Math.min(img.offsetWidth - cropData.x, dragState.startCropWidth + dx));
                    cropData.height = Math.max(50, Math.min(img.offsetHeight - cropData.y, dragState.startCropHeight + dy));
                    break;
                
                case 'tm':
                    cropData.y = Math.max(0, Math.min(dragState.startCropY + dragState.startCropHeight - 50, dragState.startCropY + dy));
                    cropData.height = dragState.startCropHeight - (cropData.y - dragState.startCropY);
                    break;
                
                case 'bm':
                    cropData.height = Math.max(50, Math.min(img.offsetHeight - cropData.y, dragState.startCropHeight + dy));
                    break;
                
                case 'ml':
                    cropData.x = Math.max(0, Math.min(dragState.startCropX + dragState.startCropWidth - 50, dragState.startCropX + dx));
                    cropData.width = dragState.startCropWidth - (cropData.x - dragState.startCropX);
                    break;
                
                case 'mr':
                    cropData.width = Math.max(50, Math.min(img.offsetWidth - cropData.x, dragState.startCropWidth + dx));
                    break;
            }
            
            updateCropBox();
        }
        
        function stopDrag() {
            if (dragState.active) {
                dragState.active = false;
                document.body.style.cursor = '';
                updatePreview();
            }
        }
        
        function updateCropBox() {
            const cropBox = document.getElementById('cropBox');
            cropBox.style.left = cropData.x + 'px';
            cropBox.style.top = cropData.y + 'px';
            cropBox.style.width = cropData.width + 'px';
            cropBox.style.height = cropData.height + 'px';
        }
        
        function handlePresetChange() {
            const preset = document.getElementById('presetSize').value;
            const customSize = document.getElementById('customSize');
            
            if (preset === '0x0') {
                customSize.style.display = 'block';
            } else {
                customSize.style.display = 'none';
                const [width, height] = preset.split('x').map(Number);
                document.getElementById('customWidth').value = width;
                document.getElementById('customHeight').value = height;
            }
            
            updatePreview();
        }
        
        function showRateLimitError(seconds, message = null) {
            const modal = document.getElementById('rateLimitError');
            const overlay = document.getElementById('rateLimitOverlay');
            const timer = document.getElementById('countdownTimer');
            const messageEl = document.getElementById('rateLimitMessage');
            const progressCircle = document.getElementById('progressCircle');
            
            // Show overlay and modal with animation
            overlay.classList.add('active');
            setTimeout(() => {
                modal.classList.add('active');
            }, 100);
            
            document.getElementById('loading').classList.remove('active');
            
            // Update message if provided
            if (message) {
                // Extract the type of limit from the message
                if (message.includes('hourly limit')) {
                    messageEl.innerHTML = '📊 <strong>Hourly Limit Reached!</strong><br>You\'ve processed 5 images this hour. That\'s our maximum to ensure quality service for everyone.';
                } else if (message.includes('processing limit') || message.includes('too many requests this minute')) {
                    messageEl.innerHTML = '⚡ <strong>Slow Down!</strong><br>You can only process 2 images per minute. This helps us maintain fast processing speeds.';
                } else {
                    messageEl.innerHTML = '⏱️ <strong>Rate Limit Active</strong><br>Please wait for the timer to complete before processing another image.';
                }
            }
            
            let remainingSeconds = seconds;
            const totalSeconds = seconds;
            
            // Setup progress circle
            const radius = 54;
            const circumference = 2 * Math.PI * radius;
            progressCircle.style.strokeDasharray = circumference;
            progressCircle.style.strokeDashoffset = 0;
            
            // Clear any existing interval
            if (countdownInterval) {
                clearInterval(countdownInterval);
            }
            
            // Update timer immediately
            updateTimerDisplay(remainingSeconds);
            updateProgress(remainingSeconds, totalSeconds);
            
            // Start countdown
            countdownInterval = setInterval(() => {
                remainingSeconds--;
                updateTimerDisplay(remainingSeconds);
                updateProgress(remainingSeconds, totalSeconds);
                
                if (remainingSeconds <= 0) {
                    clearInterval(countdownInterval);
                    closeRateLimitModal();
                    // Show success notification
                    showNotification('✅ Rate limit reset! You can now process images again.');
                }
            }, 1000);
        }
        
        function updateTimerDisplay(seconds) {
            const timer = document.getElementById('countdownTimer');
            const minutes = Math.floor(seconds / 60);
            const secs = seconds % 60;
            
            // Update with animation
            const spans = timer.querySelectorAll('span');
            spans[0].textContent = minutes.toString().padStart(2, '0');
            spans[1].textContent = secs.toString().padStart(2, '0');
            
            // Add pulse effect on each second
            timer.style.transform = 'scale(1.1)';
            setTimeout(() => {
                timer.style.transform = 'scale(1)';
            }, 200);
        }
        
        function updateProgress(current, total) {
            const progressCircle = document.getElementById('progressCircle');
            const radius = 54;
            const circumference = 2 * Math.PI * radius;
            const progress = current / total;
            const offset = circumference * (1 - progress);
            
            progressCircle.style.strokeDashoffset = offset;
        }
        
        function closeRateLimitModal() {
            const modal = document.getElementById('rateLimitError');
            const overlay = document.getElementById('rateLimitOverlay');
            
            modal.classList.remove('active');
            setTimeout(() => {
                overlay.classList.remove('active');
            }, 300);
            
            if (countdownInterval) {
                clearInterval(countdownInterval);
            }
        }
        
        function showNotification(message) {
            // Create notification element
            const notification = document.createElement('div');
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                padding: 20px 30px;
                border-radius: 12px;
                box-shadow: 0 10px 30px rgba(0,0,0,0.3);
                z-index: 2000;
                animation: slideInRight 0.5s ease;
                font-weight: 600;
            `;
            notification.textContent = message;
            
            document.body.appendChild(notification);
            
            // Remove after 3 seconds
            setTimeout(() => {
                notification.style.animation = 'slideOutRight 0.5s ease';
                setTimeout(() => {
                    document.body.removeChild(notification);
                }, 500);
            }, 3000);
        }
        
        // Add keyframes for notifications
        if (!document.getElementById('notificationStyles')) {
            const style = document.createElement('style');
            style.id = 'notificationStyles';
            style.textContent = `
                @keyframes slideInRight {
                    from { transform: translateX(100%); opacity: 0; }
                    to { transform: translateX(0); opacity: 1; }
                }
                @keyframes slideOutRight {
                    from { transform: translateX(0); opacity: 1; }
                    to { transform: translateX(100%); opacity: 0; }
                }
            `;
            document.head.appendChild(style);
        }
        
        function getTargetDimensions() {
            const preset = document.getElementById('presetSize').value;
            
            if (preset === '0x0') {
                return {
                    width: parseInt(document.getElementById('customWidth').value),
                    height: parseInt(document.getElementById('customHeight').value)
                };
            } else {
                const [width, height] = preset.split('x').map(Number);
                return { width, height };
            }
        }
        
        function updatePreview() {
            const target = getTargetDimensions();
            const brightness = document.getElementById('brightness').value;
            const contrast = document.getElementById('contrast').value;
            const quality = document.getElementById('quality').value;
            
            console.log('Updating preview with:', {
                target,
                cropData,
                imageData,
                currentBackground,
                brightness,
                contrast,
                quality
            });
            
            document.getElementById('loading').classList.add('active');
            
            // Convert crop coordinates to actual image coordinates
            const realCrop = {
                x: Math.round(cropData.x * imageData.scale),
                y: Math.round(cropData.y * imageData.scale),
                width: Math.round(cropData.width * imageData.scale),
                height: Math.round(cropData.height * imageData.scale)
            };
            
            fetch('process_image.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    width: target.width,
                    height: target.height,
                    cropX: realCrop.x,
                    cropY: realCrop.y,
                    cropWidth: realCrop.width,
                    cropHeight: realCrop.height,
                    brightness: brightness,
                    contrast: contrast,
                    background: currentBackground,
                    quality: quality
                })
            })
            .then(response => {
                console.log('Response status:', response.status);
                if (!response.ok) {
                    throw new Error('HTTP error! status: ' + response.status);
                }
                return response.text();
            })
            .then(text => {
                console.log('Response text:', text);
                try {
                    return JSON.parse(text);
                } catch (e) {
                    console.error('JSON parse error:', e);
                    console.error('Response was:', text);
                    throw new Error('Invalid JSON response. Response: ' + text.substring(0, 200));
                }
            })
            .then(data => {
                document.getElementById('loading').classList.remove('active');
                
                if (data.error) {
                    console.error('Server error:', data.error);
                    if (data.debug) {
                        console.error('Debug info:', data.debug);
                    }
                    
                    // Check if it's a rate limit error
                    if (data.error.includes('reached your') || data.error.includes('too many requests') || 
                        data.error.includes('limit') || data.error.includes('wait')) {
                        // Extract seconds from error message - looking for patterns like "wait X seconds"
                        const match = data.error.match(/wait (\d+) seconds|Remaining time: (\d+) seconds/);
                        if (match) {
                            const seconds = parseInt(match[1] || match[2]);
                            showRateLimitError(seconds, data.error);
                        } else {
                            // Default to 60 seconds if we can't parse the time
                            showRateLimitError(60, data.error);
                        }
                    } else {
                        alert('Error: ' + data.error);
                    }
                    return;
                }
                
                if (data.success) {
                    document.getElementById('preview').src = data.data;
                    document.getElementById('previewSize').textContent = target.width + '×' + target.height + 'px';
                    document.getElementById('fileSize').textContent = data.size;
                    
                    // Update size indicator
                    const indicator = document.getElementById('sizeIndicator');
                    indicator.className = 'size-indicator';
                    if (data.size <= 100) {
                        indicator.classList.add('size-optimal');
                    } else if (data.size <= 300) {
                        indicator.classList.add('size-acceptable');
                    } else {
                        indicator.classList.add('size-large');
                    }
                }
            })
            .catch(error => {
                document.getElementById('loading').classList.remove('active');
                console.error('Error:', error);
                alert('Error processing image: ' + error.message);
            });
        }
        
        function resetSettings() {
            // Reset crop to full image
            initializeCrop();
            updateCropBox();
            
            // Reset controls
            document.getElementById('presetSize').value = '413x531';
            document.getElementById('customSize').style.display = 'none';
            document.getElementById('brightness').value = 0;
            document.getElementById('contrast').value = 0;
            document.getElementById('quality').value = 85;
            
            // Reset background
            document.querySelectorAll('.bg-option').forEach(o => o.classList.remove('active'));
            document.querySelector('[data-bg="none"]').classList.add('active');
            currentBackground = 'none';
            
            // Update displays
            document.getElementById('brightnessValue').textContent = '0';
            document.getElementById('contrastValue').textContent = '0';
            document.getElementById('qualityValue').textContent = '85';
            
            updatePreview();
        }
        
        function downloadImage() {
            const preview = document.getElementById('preview');
            
            // Check if preview has a valid image
            if (!preview.src || preview.src.includes('svg+xml')) {
                alert('Please generate a preview first');
                return;
            }
            
            // Get current values
            const target = getTargetDimensions();
            const quality = document.getElementById('quality').value;
            const fileSize = document.getElementById('fileSize').textContent;
            
            // Store processed image data in session via AJAX
            fetch('store_processed_image.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    data: preview.src,
                    width: target.width,
                    height: target.height,
                    size: fileSize,
                    quality: quality
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Redirect to download page
                    window.location.href = 'download_ready.php';
                } else {
                    alert('Error preparing download: ' + (data.error || 'Unknown error'));
                }
            })
            .catch(err => {
                console.error('Error:', err);
                alert('Error preparing download');
            });
        }
    </script>
</body>
</html>