<?php
class ServerMonitor {
    private $logFile;
    private $alertThresholds;
    
    public function __construct() {
        $this->logFile = __DIR__ . '/../temp/server_stats.log';
        $this->alertThresholds = [
            'memory_percent' => 80,
            'cpu_percent' => 75,
            'disk_percent' => 90
        ];
    }
    
    public function checkResources() {
        $stats = [
            'timestamp' => time(),
            'memory' => $this->getMemoryUsage(),
            'cpu' => $this->getCpuUsage(),
            'disk' => $this->getDiskUsage(),
            'active_sessions' => $this->getActiveSessions(),
            'temp_files' => $this->getTempFileCount()
        ];
        
        // Log stats
        $this->logStats($stats);
        
        // Check if we should allow processing
        return $this->isHealthy($stats);
    }
    
    public function getMemoryUsage() {
        $memInfo = [];
        
        // Get PHP memory usage
        $memInfo['php_current'] = memory_get_usage(true);
        $memInfo['php_peak'] = memory_get_peak_usage(true);
        
        // Get system memory (Linux)
        if (file_exists('/proc/meminfo')) {
            $data = file_get_contents('/proc/meminfo');
            preg_match('/MemTotal:\s+(\d+)/', $data, $matches);
            $memInfo['total'] = isset($matches[1]) ? $matches[1] * 1024 : 0;
            
            preg_match('/MemAvailable:\s+(\d+)/', $data, $matches);
            $memInfo['available'] = isset($matches[1]) ? $matches[1] * 1024 : 0;
            
            if ($memInfo['total'] > 0) {
                $memInfo['percent_used'] = round((($memInfo['total'] - $memInfo['available']) / $memInfo['total']) * 100, 2);
            }
        }
        
        return $memInfo;
    }
    
    public function getCpuUsage() {
        $cpuInfo = [];
        
        // Get load average
        $loadAvg = sys_getloadavg();
        $cpuInfo['load_1min'] = $loadAvg[0];
        $cpuInfo['load_5min'] = $loadAvg[1];
        $cpuInfo['load_15min'] = $loadAvg[2];
        
        // Estimate CPU usage based on load and CPU count
        $cpuCount = $this->getCpuCount();
        $cpuInfo['cpu_count'] = $cpuCount;
        $cpuInfo['percent_used'] = round(($loadAvg[0] / $cpuCount) * 100, 2);
        
        return $cpuInfo;
    }
    
    public function getDiskUsage() {
        $diskInfo = [];
        
        $path = __DIR__ . '/..';
        $diskInfo['total'] = disk_total_space($path);
        $diskInfo['free'] = disk_free_space($path);
        $diskInfo['used'] = $diskInfo['total'] - $diskInfo['free'];
        $diskInfo['percent_used'] = round(($diskInfo['used'] / $diskInfo['total']) * 100, 2);
        
        return $diskInfo;
    }
    
    public function getActiveSessions() {
        $sessionPath = session_save_path();
        if (empty($sessionPath)) {
            $sessionPath = sys_get_temp_dir();
        }
        
        $sessionFiles = glob($sessionPath . '/sess_*');
        $activeSessions = 0;
        $recentThreshold = time() - 300; // 5 minutes
        
        foreach ($sessionFiles as $file) {
            if (filemtime($file) > $recentThreshold) {
                $activeSessions++;
            }
        }
        
        return $activeSessions;
    }
    
    public function getTempFileCount() {
        $tempDir = __DIR__ . '/../temp/';
        $files = glob($tempDir . '*');
        return count($files);
    }
    
    private function getCpuCount() {
        $cpuCount = 1;
        
        if (file_exists('/proc/cpuinfo')) {
            $cpuInfo = file_get_contents('/proc/cpuinfo');
            preg_match_all('/processor\s+:/', $cpuInfo, $matches);
            $cpuCount = count($matches[0]);
        }
        
        return $cpuCount;
    }
    
    private function isHealthy($stats) {
        // Check memory
        if (isset($stats['memory']['percent_used']) && 
            $stats['memory']['percent_used'] > $this->alertThresholds['memory_percent']) {
            return false;
        }
        
        // Check CPU
        if (isset($stats['cpu']['percent_used']) && 
            $stats['cpu']['percent_used'] > $this->alertThresholds['cpu_percent']) {
            return false;
        }
        
        // Check disk
        if (isset($stats['disk']['percent_used']) && 
            $stats['disk']['percent_used'] > $this->alertThresholds['disk_percent']) {
            return false;
        }
        
        return true;
    }
    
    private function logStats($stats) {
        $logEntry = date('Y-m-d H:i:s') . ' | ' . json_encode($stats) . PHP_EOL;
        
        // Rotate log if too large (> 10MB)
        if (file_exists($this->logFile) && filesize($this->logFile) > 10485760) {
            rename($this->logFile, $this->logFile . '.' . date('Ymd'));
        }
        
        file_put_contents($this->logFile, $logEntry, FILE_APPEND | LOCK_EX);
    }
    
    public function getRecentStats($minutes = 60) {
        if (!file_exists($this->logFile)) {
            return [];
        }
        
        $stats = [];
        $lines = file($this->logFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $cutoff = time() - ($minutes * 60);
        
        foreach (array_reverse($lines) as $line) {
            if (preg_match('/^(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}) \| (.+)$/', $line, $matches)) {
                $timestamp = strtotime($matches[1]);
                if ($timestamp < $cutoff) {
                    break;
                }
                
                $data = json_decode($matches[2], true);
                if ($data) {
                    $stats[] = $data;
                }
            }
        }
        
        return array_reverse($stats);
    }
}
?>