<?php
class RateLimiter {
    private $sessionId;
    private $cacheDir;
    private $maxRequests;
    private $timeWindow;
    
    public function __construct($maxRequests = 10, $timeWindow = 60) {
        $this->sessionId = session_id();
        $this->cacheDir = __DIR__ . '/../temp/rate_limit/';
        $this->maxRequests = $maxRequests;
        $this->timeWindow = $timeWindow; // in seconds
        
        // Create cache directory if it doesn't exist
        if (!file_exists($this->cacheDir)) {
            mkdir($this->cacheDir, 0777, true);
        }
        
        // Clean old rate limit files
        $this->cleanOldFiles();
    }
    
    public function isAllowed($action = 'default') {
        $file = $this->getCacheFile($action);
        $requests = $this->getRequests($file);
        $now = time();
        
        // Remove old requests outside the time window
        $requests = array_filter($requests, function($timestamp) use ($now) {
            return ($now - $timestamp) < $this->timeWindow;
        });
        
        // Check if limit exceeded
        if (count($requests) >= $this->maxRequests) {
            return false;
        }
        
        // Add current request
        $requests[] = $now;
        $this->saveRequests($file, $requests);
        
        return true;
    }
    
    public function getRemainingRequests($action = 'default') {
        $file = $this->getCacheFile($action);
        $requests = $this->getRequests($file);
        $now = time();
        
        // Count requests within time window
        $recentRequests = array_filter($requests, function($timestamp) use ($now) {
            return ($now - $timestamp) < $this->timeWindow;
        });
        
        return max(0, $this->maxRequests - count($recentRequests));
    }
    
    public function getResetTime($action = 'default') {
        $file = $this->getCacheFile($action);
        $requests = $this->getRequests($file);
        
        if (empty($requests)) {
            return 0;
        }
        
        // Find oldest request within window
        $oldestRequest = min($requests);
        return $oldestRequest + $this->timeWindow;
    }
    
    public function getWaitTime($action = 'default') {
        $file = $this->getCacheFile($action);
        $requests = $this->getRequests($file);
        $now = time();
        
        // Filter requests within time window
        $recentRequests = array_filter($requests, function($timestamp) use ($now) {
            return ($now - $timestamp) < $this->timeWindow;
        });
        
        if (count($recentRequests) < $this->maxRequests) {
            return 0;
        }
        
        // Find the oldest request that will expire
        $oldestRequest = min($recentRequests);
        $waitTime = ($oldestRequest + $this->timeWindow) - $now;
        
        return max(0, $waitTime);
    }
    
    private function getCacheFile($action) {
        return $this->cacheDir . md5($this->sessionId . '_' . $action) . '.json';
    }
    
    private function getRequests($file) {
        if (!file_exists($file)) {
            return [];
        }
        
        $data = json_decode(file_get_contents($file), true);
        return is_array($data) ? $data : [];
    }
    
    private function saveRequests($file, $requests) {
        file_put_contents($file, json_encode(array_values($requests)));
    }
    
    private function cleanOldFiles() {
        $files = glob($this->cacheDir . '*.json');
        $now = time();
        
        foreach ($files as $file) {
            // Remove files older than 1 hour
            if ($now - filemtime($file) > 3600) {
                @unlink($file);
            }
        }
    }
}
?>