<?php
class IPRateLimiter {
    private $cacheDir;
    private $minuteLimit;
    private $hourLimit;
    
    public function __construct($minuteLimit = 2, $hourLimit = 5) {
        $this->cacheDir = __DIR__ . '/../temp/ip_rate_limit/';
        $this->minuteLimit = $minuteLimit;
        $this->hourLimit = $hourLimit;
        
        // Create cache directory if it doesn't exist
        if (!file_exists($this->cacheDir)) {
            mkdir($this->cacheDir, 0777, true);
        }
        
        // Clean old files
        $this->cleanOldFiles();
    }
    
    public function getUserIP() {
        // Get real IP even behind proxy/cloudflare
        $ip_keys = ['HTTP_CF_CONNECTING_IP', 'HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR'];
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                $ip = trim($_SERVER[$key]);
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                    return $ip;
                }
            }
        }
        return $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
    }
    
    public function isAllowed() {
        $ip = $this->getUserIP();
        $now = time();
        
        // Check minute limit
        if (!$this->checkLimit($ip, 'minute', 60, $this->minuteLimit)) {
            return ['allowed' => false, 'reason' => 'You\'ve made too many requests this minute (limit: ' . $this->minuteLimit . ' per minute).'];
        }
        
        // Check hour limit
        if (!$this->checkLimit($ip, 'hour', 3600, $this->hourLimit)) {
            return ['allowed' => false, 'reason' => 'You\'ve reached your hourly limit (limit: ' . $this->hourLimit . ' per hour).'];
        }
        
        // Record this request
        $this->recordRequest($ip);
        
        return ['allowed' => true];
    }
    
    private function checkLimit($ip, $period, $seconds, $limit) {
        $file = $this->getCacheFile($ip);
        $requests = $this->getRequests($file);
        $now = time();
        
        // Count requests within the time window
        $count = 0;
        foreach ($requests as $timestamp) {
            if (($now - $timestamp) < $seconds) {
                $count++;
            }
        }
        
        return $count < $limit;
    }
    
    private function recordRequest($ip) {
        $file = $this->getCacheFile($ip);
        $requests = $this->getRequests($file);
        $now = time();
        
        // Add current request
        $requests[] = $now;
        
        // Keep only requests from last hour
        $requests = array_filter($requests, function($timestamp) use ($now) {
            return ($now - $timestamp) < 3600;
        });
        
        $this->saveRequests($file, $requests);
    }
    
    public function getRemainingRequests() {
        $ip = $this->getUserIP();
        $file = $this->getCacheFile($ip);
        $requests = $this->getRequests($file);
        $now = time();
        
        // Count minute requests
        $minuteCount = 0;
        $hourCount = 0;
        
        foreach ($requests as $timestamp) {
            if (($now - $timestamp) < 60) {
                $minuteCount++;
            }
            if (($now - $timestamp) < 3600) {
                $hourCount++;
            }
        }
        
        return [
            'minute' => max(0, $this->minuteLimit - $minuteCount),
            'hour' => max(0, $this->hourLimit - $hourCount)
        ];
    }
    
    public function getWaitTime() {
        $ip = $this->getUserIP();
        $file = $this->getCacheFile($ip);
        $requests = $this->getRequests($file);
        $now = time();
        
        // Filter requests for minute and hour windows
        $minuteRequests = [];
        $hourRequests = [];
        
        foreach ($requests as $timestamp) {
            if (($now - $timestamp) < 60) {
                $minuteRequests[] = $timestamp;
            }
            if (($now - $timestamp) < 3600) {
                $hourRequests[] = $timestamp;
            }
        }
        
        $waitTime = 0;
        
        // Check minute limit
        if (count($minuteRequests) >= $this->minuteLimit) {
            $oldestMinuteRequest = min($minuteRequests);
            $minuteWait = ($oldestMinuteRequest + 60) - $now;
            $waitTime = max($waitTime, $minuteWait);
        }
        
        // Check hour limit
        if (count($hourRequests) >= $this->hourLimit) {
            $oldestHourRequest = min($hourRequests);
            $hourWait = ($oldestHourRequest + 3600) - $now;
            $waitTime = max($waitTime, $hourWait);
        }
        
        return max(0, $waitTime);
    }
    
    private function getCacheFile($ip) {
        // Hash IP for privacy
        return $this->cacheDir . md5($ip . '_rate_limit') . '.json';
    }
    
    private function getRequests($file) {
        if (!file_exists($file)) {
            return [];
        }
        
        $data = json_decode(file_get_contents($file), true);
        return is_array($data) ? $data : [];
    }
    
    private function saveRequests($file, $requests) {
        file_put_contents($file, json_encode(array_values($requests)), LOCK_EX);
    }
    
    private function cleanOldFiles() {
        $files = glob($this->cacheDir . '*.json');
        $now = time();
        
        foreach ($files as $file) {
            // Remove files older than 2 hours
            if ($now - filemtime($file) > 7200) {
                @unlink($file);
            }
        }
    }
    
    // Check if IP is suspicious (rapid requests)
    public function isSuspicious() {
        $ip = $this->getUserIP();
        $file = $this->getCacheFile($ip);
        $requests = $this->getRequests($file);
        $now = time();
        
        // Count requests in last 10 seconds
        $recentCount = 0;
        foreach ($requests as $timestamp) {
            if (($now - $timestamp) < 10) {
                $recentCount++;
            }
        }
        
        // Suspicious if more than 3 requests in 10 seconds
        return $recentCount > 3;
    }
}
?>